<?php

namespace Drupal\monster_menus\MMSearchAction;

use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

abstract class MMSearchActionBase implements MMSearchActionInterface, FormInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /*
   * @var AccountInterface
   */
  public $currentUser;

  /**
   * @var PagerManagerInterface
   */
  protected $pagerManager;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  public $updateNeeded, $values;

  final public const string SEARCH_TYPE_NODES = 'nodes';
  final public const string SEARCH_TYPE_PAGES = 'pages';
  final public const string SEARCH_TYPE_NODES_ON_PAGES = 'nodes+pages';
  final public const string SEARCH_TYPE_GROUPS = 'groups';

  public function __construct(public $configuration, protected $pluginId, AccountInterface $current_user, PagerManagerInterface $pager_manager, Connection $database) {
    $this->currentUser = $current_user;
    $this->pagerManager = $pager_manager;
    $this->database = $database;
  }

  /**
   * @inheritDoc
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $container->get('current_user'),
      $container->get('pager.manager'),
      $container->get('database'),
    );
  }

  public function getConfiguration($option = NULL) {
    return isset($option) ? ($this->configuration[$option] ?? NULL) : $this->configuration;
  }

  public function addStatusMarkup(&$form) {
    $form['actions']['status'] = [
      '#markup' => '<div id="mmsr-mp-status" class="clearfix"><div id="mmsr-mp-status-text"></div><div class="ajax-progress ajax-progress-throbber"><div class="throbber">&nbsp;&nbsp;</div></div></div>',
    ];
  }

  /**
   * @inheritDoc
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   * @inheritDoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

  public function setSubmittedValues(FormStateInterface $form_state) {
    // Prevent an attempt to GET the page again afterward.
    $form_state->disableRedirect();
    $this->updateNeeded = $form_state->getTriggeringElement()['#id'] != 'edit-calc';
    $this->values = $form_state->getValues();
  }
  
  public function iterate($node_func, $page_func, $per_page = 100, $use_pager = FALSE) {
    $offset = 0;
    if ($use_pager) {
      $pager = $this->pagerManager->createPager($this->getConfiguration('result_count'), $per_page);
      $offset = $per_page * $pager->getCurrentPage();
    }
    // Run in batches, so the underlying database code doesn't try to read
    // all rows into memory.
    for (;;) {
      $result = $this->database->queryRange($this->getConfiguration('result_query'), $offset, $per_page);
      $had_row = FALSE;
      foreach ($result as $row) {
        $had_row = TRUE;
        if (isset($row->nid)) {
          $return = $node_func($row);
        }
        else if (isset($row->mmtid)) {
          $return = $page_func($row);
        }
        if (isset($return)) {
          return $return;
        }
      }
      if (!$had_row || $use_pager) {
        return;
      }
      $offset += $per_page;
    }
  }

  /**
   * @inheritDoc
   */
  public function getFormId() {
    return $this->pluginId;
  }

  /**
   * @inheritDoc
   */
  public function applies() {
    return FALSE;
  }

  /**
   * @inheritDoc
   */
  public function access() {
    return FALSE;
  }

}