<?php

namespace Drupal\monster_menus\Menu;

use Drupal\Core\Access\AccessManagerInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Menu\DefaultMenuLinkTreeManipulators;
use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\Routing\Route;

/**
 * Overrides DefaultMenuLinkTreeManipulators to allow for menu links which
 * contain unspecified {mm_tree} parameters.
 */
class MenuLinkTreeManipulators extends DefaultMenuLinkTreeManipulators {

  /**
   * @inheritDoc
   */
  public function __construct(AccessManagerInterface $accessManager, AccountInterface $account, EntityTypeManagerInterface $entityTypeManager, ModuleHandlerInterface $moduleHandler, protected RouteProviderInterface $routeProvider) {
    parent::__construct($accessManager, $account, $entityTypeManager, $moduleHandler);
  }

  /**
   * Checks access for one menu link instance. We need to override this because
   * menus can refer to paths containing {mm_tree}, which might not be supplied
   * as a parameter.
   *
   * @param MenuLinkInterface $instance
   *   The menu link instance.
   *
   * @return AccessResultInterface
   *   The access result.
   */
  protected function menuLinkCheckAccess(MenuLinkInterface $instance) {
    if ($this->account->hasPermission('link to any page')) {
      $access_result = AccessResult::allowed();
    }
    else {
      $url = $instance->getUrlObject();

      // When no route name is specified, this must be an external link.
      if (!$url->isRouted()) {
        $access_result = AccessResult::allowed();
      }
      else {
        /** @var Route $route */
        $route = $this->routeProvider->getRouteByName($url->getRouteName());
        // If {mm_tree} is present in the path but not supplied, use the current
        // page or the homepage.
        if (isset($route->getOption('parameters')['mm_tree']) && !isset($url->getRouteParameters()['mm_tree'])) {
          mm_parse_args($mmtids, $oarg_list, $this_mmtid);
          $url->setRouteParameter('mm_tree', $this_mmtid ?: mm_home_mmtid());
        }
        $access_result = $this->accessManager->checkNamedRoute($url->getRouteName(), $url->getRouteParameters(), $this->account, TRUE);
      }
    }
    /** @var AccessResult $access_result */
    return $access_result->cachePerPermissions();
  }

}
