<?php

namespace Drupal\monster_menus\Plugin\MMSearchAction;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\monster_menus\Annotation\MMSearchAction;
use Drupal\monster_menus\Controller\DefaultController;
use Drupal\monster_menus\MMSearchAction\MMSearchActionBase;

/**
 * Provides the MM Search Results Action to display results in a table.
 *
 * @MMSearchAction(
 *   id = "mm_search_action_display_table",
 *   label = @Translation("display results in a table"),
 *   description = @Translation("Provides the MM Search Results Action to display results in a table."),
 *   jsInit = "MMSR_init_action_pager",
 * )
 */
class ResultsTable extends MMSearchActionBase {

  final public const int NUM_PER_PAGE = 10;

  /**
   * @inheritDoc
   * @return mixed[]
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // With cache enabled paging and column sorting get messed up.
    $form_state->disableCache();
    if ($form_state->getValue('op')) {
      return [
        $this->showTable(),
        'op' => [
          '#type' => 'hidden',
          '#value' => $form_state->getValue('op'),
        ],
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      'result' => [
        '#type' => 'button',
        '#value' => $this->t('Display Results'),
        '#attributes' => ['onclick' => 'MMSR_recalculate_action(this); return false;'],
      ],
    ];
    return $form;
  }

  public function showTable() {
    $rows = [];
    $this->iterate(function ($row) use (&$rows) {
      $rows[] = [
        'data' => [
          Link::fromTextAndUrl(trim($row->title) == '' ? $this->t('(unknown)') : $row->title, Url::fromRoute('entity.node.canonical', ['node' => $row->nid]))
            ->toString(),
          DefaultController::getNodeTypeLabel($row->type),
          mm_format_date($row->changed, 'short'),
          mm_format_date($row->created, 'short'),
          mm_content_uid2name($row->uid, 'fmlu', $row, $hover),
        ],
      ];
    }, function ($row) use (&$rows) {
      $rows[] = [
        'data' => [
          Link::fromTextAndUrl(trim($row->pgname) == '' ? $this->t('(unknown)') : $row->pgname, mm_content_get_mmtid_url($row->mmtid)),
          mm_content_uid2name($row->uid, 'fmlu', $row, $hover),
        ],
      ];
    }, self::NUM_PER_PAGE, TRUE);

    if (!$rows) {
      $rows[] = [['data' => $this->t('No matches'), 'colspan' => count($this->getConfiguration('header'))]];
    }

    return [
      [
        '#type' => 'table',
        '#header' => $this->getConfiguration('header'),
        '#rows' => $rows,
      ], [
        '#type' => 'pager',
      ],
    ];
  }

  /**
   * @inheritDoc
   */
  public function applies() {
    return TRUE;
  }

  /**
   * @inheritDoc
   */
  public function access() {
    return TRUE;
  }

}