<?php

namespace Drupal\monster_menus\Plugin\MMTreeBrowserDisplay;

use Drupal\Core\Render\Markup;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\monster_menus\Constants;
use Drupal\monster_menus\MMTreeBrowserDisplay\MMTreeBrowserDisplayInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * Provides the MM Tree display generator for MMTree entities.
 *
 * @MMTreeBrowserDisplay(
 *   id = "mm_tree_browser_display_default",
 *   admin_label = @Translation("MM Tree default display"),
 * )
 */
class Fallback implements MMTreeBrowserDisplayInterface {

  final public const string BROWSER_MODE_PAGE = 'pag';
  final public const string BROWSER_MODE_ADMIN_PAGE = 'apg';

  private $searchActionRef;

  public function __construct(protected AccountInterface $currentUser) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static($container->get('current_user'));
  }

  /**
   * {@inheritdoc}
   */
  public static function supportedModes() {
    return [self::BROWSER_MODE_PAGE, self::BROWSER_MODE_ADMIN_PAGE];
  }

  /**
   * @inheritDoc
   */
  public function label($mode) {
    return $mode == self::BROWSER_MODE_ADMIN_PAGE ? t('Browse pages') : t('Select a page');
  }

  /**
   * @inheritDoc
   */
  public function showReservedEntries($mode) {
    return $this->currentUser->hasPermission('administer all menus') || $this->currentUser->hasPermission('view all menus');
  }

  /**
   * {@inheritdoc}
   */
  public function alterLeftQuery($mode, $query, &$params) {
    $params[Constants::MM_GET_TREE_FILTER_NORMAL] = $params[Constants::MM_GET_TREE_FILTER_USERS] = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function alterRightButtons($mode, $query, $item, $permissions, &$actions, &$dialogs) {
    if ($mode == self::BROWSER_MODE_ADMIN_PAGE) {
      $this->adminLinks($mode, $query, $item, $permissions, $actions);
    }
    else if ($mode == self::BROWSER_MODE_PAGE) {
      $this->selectLink($mode, $query, $item, $permissions, $actions);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function viewRight($mode, $query, $perms, $item, $database) {
    $choice = function($item, $cat, $usr, $group) {
      if (!empty($item->is_user)) {
        return $usr;
      }
      if (!empty($item->is_group)) {
        return $group;
      }
      return $cat;
    };

    $content = '';
    if (empty($item->is_virtual)) {
      $rows = [];
      $content = [
        '#type' => 'table',
        '#attributes' => ['class' => 'mmtree-browse-group'],
      ];
      if (empty($item->is_group)) {
        $rows[] = [t('URL alias:'), ($item->alias ?? '')];
      }
      if (isset($item->nodecount)) {
        $rows[] = [t('Items on page:'), $item->nodecount];
      }
      $rows[] = [t('Owner:'), mm_content_uid2name($item->uid)];
      $can = [];
      $avail_perms = [
        Constants::MM_PERMS_WRITE => $choice($item,
          t('delete/edit page'),
          t('delete/edit user'),
          t('delete/edit group')),
        Constants::MM_PERMS_SUB => $choice($item,
          t('add sub-pages'),
          $item->name == Constants::MM_ENTRY_NAME_USERS ? t('add users') : t('add sub-pages'),
          t('add sub-groups')),
        Constants::MM_PERMS_APPLY => !empty($item->is_group) ? t('apply this group') : t('assign content to page'),
        Constants::MM_PERMS_READ => !empty($item->is_group) ? t('see group\'s members') : t('read contents of page'),
      ];
      foreach ($avail_perms as $type => $desc) {
        if ($perms[$type]) {
          $can[] = $desc;
        }
      }
      $rows[] = [t('You can:'), Markup::create(implode('<br />', $can))];
      if ($item->is_group && ($users = $this->getUsersInGroup($item, $content))) {
        $rows[] = [t('Users in group:'), Markup::create($users)];
      }
      $content['#rows'] = $rows;
    }

    return $content;
  }

  /**
   * {@inheritdoc}
   */
  public function getTreeTop($mode) {
    return 1;
  }

  /**
   * {@inheritdoc}
   */
  public function getBookmarksType($mode) {
    return 'browser';
  }

  /**
   * Generate buttons specific to the /admin/mm pages.
   *
   * @param $mode
   *   Display mode constant.
   * @param ParameterBag $query
   *   The request query object.
   * @param $item
   *   Object describing the tree entry.
   * @param array $permissions
   *   The item's permissions.
   * @param $actions
   *   Array of action buttons to alter.
   */
  public function adminLinks($mode, $query, $item, $permissions, &$actions) {
    $x = mm_ui_strings($item->is_group);

    if ($permissions[Constants::MM_PERMS_READ]) {
      $actions['contents'] = [
        '#type' => 'button',
        '#value' => t('View Contents'),
        '#attributes' => [
          'title' => t('View this @thingpos contents', $x),
          'onclick' => "location.href='" . mm_content_get_mmtid_url($item->mmtid)->toString() . "'",
        ],
      ];
    }

    if ($permissions[Constants::MM_PERMS_WRITE] || $permissions[Constants::MM_PERMS_SUB]) {
      $actions['settings'] = [
        '#type' => 'button',
        '#value' => t('Change Settings'),
        '#attributes' => [
          'title' => t('Edit this @thing', $x),
          'onclick' => "location.href='" . Url::fromRoute('monster_menus.handle_page_settings', ['mm_tree' => $item->mmtid])->toString() . "'",
        ],
      ];
    }
  }

  /**
   * @param $mode
   *   Display mode constant.
   * @param ParameterBag $query
   *   The request query object.
   * @param $item
   *   Object describing the tree entry.
   * @param array $permissions
   *   The item's permissions.
   * @param $actions
   *   Array of buttons to alter.
   */
  public function selectLink($mode, $query, $item, $permissions, &$actions) {
    if (empty($item->is_virtual) && static::userCan($query, $permissions)) {
      $js_parms = $item->is_group ? "'" . mm_ui_js_escape($this->getUsersInGroup($item, $actions, FALSE)) . "'" : 0;
      $actions['select'] = [
        '#type' => 'button',
        '#value' => t('Select'),
        '#attributes' => [
          'onclick' => "Drupal.mm_browser_page_add($item->mmtid, $js_parms)",
        ],
      ];
    }
  }

  private function getUsersInGroup($item, &$render_array, $see_all = TRUE) {
    $users = mm_content_get_users_in_group($item->mmtid, '<br />', FALSE, 20, $see_all, $render_array);
    if ($users == '') {
      $users = t('(none)');
    }
    return $users;
  }

  /**
   * Determine if a given tree entry should be selectable to the user.
   *
   * @param ParameterBag $query
   *   The request query object.
   * @param array $permissions
   *   Array of permissions to test.
   * @return bool
   *   TRUE if the user has any of the permissions.
   */
  public static function userCan(ParameterBag $query, $permissions) {
    $user_can = TRUE;
    if ($selectable = $query->get('browserSelectable', '')) {
      $user_can = FALSE;
      foreach (str_split($selectable) as $check) {
        if (!empty($permissions[$check])) {
          return TRUE;
        }
      }
    }
    return $user_can;
  }

  public function addSearchAction(ParameterBag $query, &$actions, $placeholder, $weight = -2) {
    $actions['search'] = [
      '#type' => 'textfield',
      '#prefix' => '<div class="container-inline">',
      '#placeholder' => $placeholder,
      '#size' => 30,
      '#weight' => $weight++,
      '#attributes' => ['id' => 'search' , 'data-node-id' => $query->get('id'), 'onkeydown' => <<<JS_WRAP
var \$ = jQuery;

Drupal.MMSearch = function () {
  Drupal.MMSearchTimeoutID = 0;
  var filter = \$('#search').val();
  if (filter !== old_filter) {
    var params = Drupal.mm_browser_params_json();
    params.id = node_id;
    params.search = filter;
    \$.getJSON(drupalSettings.path.baseUrl + 'mm-browser-getright',
      params,
      function(data) {
        \$('#mmtree-assist-content')
          .html(data.body)
          .find('a:not([onclick]):not([id^="mm-dialog"])')
            .click(Drupal.mm_browser_right_link_click);
        // Initialize any modal dialog links.
        if (typeof Drupal.mmDialogInit !== 'undefined') {
          Drupal.mmDialogInit(\$('#mmtree-assist-content,#mmtree-assist-links'), data.dialogs);
        }
      }
    );
  }
};

var node_id = \$(this).attr('data-node-id');
var old_filter = \$('#search').val();
if (Drupal.MMSearchTimeoutID) {
  clearTimeout(Drupal.MMSearchTimeoutID);
}
Drupal.MMSearchTimeoutID = setTimeout(Drupal.MMSearch, 500);
return true;
JS_WRAP

  ,
      ],
    ];
    $actions['clear'] = [
        '#type' => 'button',
        '#value' => t('Clear'),
        '#suffix' => '</div>',
        '#weight' => $weight,
        '#attributes' => ['id' => 'search' , 'data-node-id' => $query->get('id'), 'onclick' => <<<JS
jQuery('#search').val('');
Drupal.MMSearch();
return false;
JS
  ,
      ],
    ];
    $this->searchActionRef = &$actions;
  }

  public function removeSearchAction() {
    if ($this->searchActionRef) {
      unset($this->searchActionRef['search']);
      unset($this->searchActionRef['clear']);
    }
  }

  public function getSearchTerms(ParameterBag $query) {
    if ($search = trim(preg_replace('{\W}', ' ', $query->get('search', '')))) {
      return preg_split('{\s+}', $search);
    }
    return [];
  }

}
