<?php

namespace Drupal\monster_menus\Session;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\Entity\Role;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Override \Drupal\Core\Session\AccountProxy to include permissions checking
 * which takes roles tied to MM groups into account.
 */
class AccountProxy extends \Drupal\Core\Session\AccountProxy {

  /**
   * @var array Cache for user roles derived from MM groups.
   */
  private array $userRoles;

  public $user_mmtid;

  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * @inheritDoc
   */
  public function __construct(EventDispatcherInterface $event_dispatcher, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($event_dispatcher);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function hasPermission($permission) {
    if ($this->getAccount()->hasPermission($permission)) {
      return TRUE;
    }
    return $this->entityTypeManager->getStorage('user_role')->isPermissionInRoles($permission, $this->getMMRoles());
  }

  public function getMMRoles() {
    $account = $this->getAccount();
    if ($account->isAnonymous()) {
      // Save time by avoiding the query.
      return [];
    }
    $uid = $account->id();
    if (!isset($this->userRoles[$uid])) {
      $this->userRoles[$uid] = [];
      $rids = static::getRolesHavingMMGroups();
      if ($rids) {
        /** @var Role $role */
        foreach ($this->entityTypeManager->getStorage('user_role')->loadMultiple($rids) as $role) {
          $gid = $role->get('mm_gid');
          $exclude = $role->get('mm_exclude');
          if ((bool) mm_content_get_uids_in_group($gid, $uid) !== (bool) $exclude) {
            $this->userRoles[$uid][] = $role->id();
          }
        }
      }
    }
    return $this->userRoles[$uid];
  }

  public static function getRolesHavingMMGroups() {
    return \Drupal::entityQuery('user_role')
      ->accessCheck(FALSE)
      ->condition('mm_gid', 0, '>')
      ->execute();
  }

}
