<?php

namespace Drupal\Tests\monster_menus\Unit\Controller;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\monster_menus\CheckOrphanNodes;
use Drupal\monster_menus\DumpCSV;
use Drupal\monster_menus\ValidateSortIndex;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Prophecy\PhpUnit\ProphecyTrait;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Session\AccountInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\monster_menus\Controller\DefaultController;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * @coversDefaultClass \Drupal\monster_menus\Controller\DefaultController
 * @group monster_menus
 */
class DefaultControllerTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The mocked database connection.
   */
  protected MockObject $database;
  protected ContainerInterface $container;
  protected FormBuilderInterface $formBuilder;
  protected ModuleHandlerInterface $moduleHandler;
  protected RendererInterface $renderer;
  protected RequestStack $requestStack;
  protected Request $request;
  protected DumpCSV $dumpCSV;
  protected ValidateSortIndex $validateSortIndex;
  protected CheckOrphanNodes $checkOrphanNodes;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->database = $this->createMock(Connection::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->formBuilder = $this->createMock(FormBuilderInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->renderer = $this->createMock(RendererInterface::class);
    $this->request = $this->createMock(Request::class);
    $this->requestStack = new RequestStack([$this->request]);
    $this->dumpCSV = $this->createMock(DumpCSV::class);
    $this->validateSortIndex = $this->createMock(ValidateSortIndex::class);
    $this->checkOrphanNodes = $this->createMock(CheckOrphanNodes::class);

    $this->container = new \Symfony\Component\DependencyInjection\ContainerBuilder();
    $this->container->set('database', $this->database);
    $this->container->set('entity_type.manager', $this->entityTypeManager);
    $this->container->set('form_builder', $this->formBuilder);
    $this->container->set('module_handler', $this->moduleHandler);
    $this->container->set('request_stack', $this->requestStack);
    $this->container->set('renderer', $this->renderer);
    $this->container->set('monster_menus.dump_csv', $this->dumpCSV);
    $this->container->set('monster_menus.validate_sort_index', $this->validateSortIndex);
    $this->container->set('monster_menus.check_orphan_nodes', $this->checkOrphanNodes);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    parent::tearDown();
    $container = new ContainerBuilder();
    \Drupal::setContainer($container);
  }

  /**
   * Test the static create method.
   */
  public function testCreate() {
    $container = $this->createMock(ContainerInterface::class);
    $container->expects($this->any())
      ->method('get')
      ->willReturn($this->container, $this->database);

    // @phpstan-ignore method.alreadyNarrowedType
    $this->assertInstanceOf(DefaultController::class, DefaultController::create($this->container));
  }

  /**
   * Test for 'menuAccessCreateHomepage'.
   *
   * @covers ::menuAccessCreateHomepage
   *
   * @dataProvider providerTestMenuAccessCreateHomepage
   */
  /*
   * @TODO: handle mm_get_setting() so this test doesn't fail to run.
  public function testMenuAccessCreateHomepage(AccountInterface $user, User $account, AccessResult $expected_result) {

    $container = new ContainerBuilder();
    $container->set('current_user', $user);
    \Drupal::setContainer($container);

    $this->assertEquals($expected_result, DefaultController::menuAccessCreateHomepage($account));
  }
  */

  /**
   * Data provider for ::testMenuAccessCreateHomepage.
   */
  public function providerTestMenuAccessCreateHomepage() {

    $cases = [];

    // Test if the account isn't active.
    $user = $this->prophesize(AccountInterface::class);
    $user->hasPermission('administer all users')->willReturn(TRUE);
    $account = $this->prophesize(User::class);
    $account->isActive()->willReturn(FALSE);
    $cases[] = [$user->reveal(), $account->reveal(), AccessResult::neutral()];

    // Test if the account is active and the current user has admin permissions.
    $user = $this->prophesize(AccountInterface::class);
    $user->hasPermission('administer all users')->willReturn(TRUE);
    $account = $this->prophesize(User::class);
    $account->isActive()->willReturn(TRUE);
    $cases[] = [$user->reveal(), $account->reveal(), AccessResult::allowed()];

    // Test if the account is active and doesn't have admin permissions, but the ids match.
    $user = $this->prophesize(AccountInterface::class);
    $user->hasPermission('administer all users')->willReturn(FALSE);
    $user->id()->willReturn(25);
    $account = $this->prophesize(User::class);
    $account->isActive()->willReturn(TRUE);
    $account->id()->willReturn(25);
    $cases[] = [$user->reveal(), $account->reveal(), AccessResult::allowed()];

    // Test if the account is active and doesn't have admin permissions, and the ids don't match.
    $user = $this->prophesize(AccountInterface::class);
    $user->hasPermission('administer all users')->willReturn(FALSE);
    $user->id()->willReturn(26);
    $account = $this->prophesize(User::class);
    $account->isActive()->willReturn(TRUE);
    $account->id()->willReturn(25);
    $cases[] = [$user->reveal(), $account->reveal(), AccessResult::neutral()];

    return $cases;
  }

  /**
   * Test for accessAnyAdmin.
   *
   * @covers ::accessAnyAdmin
   */
  public function testAccessAnyAdmin() {
    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(TRUE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::allowed(), DefaultController::accessAnyAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(TRUE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::allowed(), DefaultController::accessAnyAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(TRUE);
    $this->assertEquals(AccessResult::allowed(), DefaultController::accessAnyAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::neutral(), DefaultController::accessAnyAdmin($account->reveal()));
  }

  /**
   * Test for accessAllAdmin.
   *
   * @covers ::accessAllAdmin
   */
  public function testAccessAllAdmin() {
    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(TRUE);
    $account->hasPermission('administer all users')->willReturn(TRUE);
    $account->hasPermission('administer all menus')->willReturn(TRUE);
    $this->assertEquals(AccessResult::allowed(), DefaultController::accessAllAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(TRUE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::neutral(), DefaultController::accessAllAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(TRUE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::neutral(), DefaultController::accessAllAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(TRUE);
    $this->assertEquals(AccessResult::neutral(), DefaultController::accessAllAdmin($account->reveal()));

    $account = $this->prophesize(AccountInterface::class);
    $account->hasPermission('administer all groups')->willReturn(FALSE);
    $account->hasPermission('administer all users')->willReturn(FALSE);
    $account->hasPermission('administer all menus')->willReturn(FALSE);
    $this->assertEquals(AccessResult::neutral(), DefaultController::accessAllAdmin($account->reveal()));
  }

}