<?php

namespace Drupal\month_year_range\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\datetime\Plugin\Field\FieldWidget\DatetimeDatelistWidget;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Plugin implementation of the 'month_year_datetime' widget.
 *
 * @FieldWidget(
 *   id = "month_year_datetime",
 *   label = @Translation("Month Year datetime"),
 *   field_types = {
 *     "datetime"
 *   }
 * )
 */
class MonthYearDatetimeWidget extends DatetimeDatelistWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'date_order' => 'YM',
        'year_range' => '',
        'day_option' => 'first',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // Éviter parent::formElement() pour contourner le problème de match
    $element['value'] = [
      '#type' => 'datelist',
      '#title' => $this->t('Date'),
      '#title_display' => 'invisible',
      '#default_value' => NULL,
      '#required' => $element['#required'],
      '#size' => max(6, strlen($this->getSetting('placeholder'))),
    ];

    // Récupérer la valeur actuelle
    if (isset($items[$delta]->value)) {
      try {
        $date = new DrupalDateTime($items[$delta]->value);
        $element['value']['#default_value'] = $date;
      } catch (\Exception $e) {
        // Ignorer les erreurs de parsing de date
      }
    }

    $date_order = $this->getSetting('date_order');
    $year_range = $this->getSetting('year_range');

    // Set up the date part order array.
    switch ($date_order) {
      default:
      case 'YM':
        $date_part_order = ['year', 'month'];
        break;

      case 'MY':
        $date_part_order = ['month', 'year'];
        break;

      case 'Y':
        $date_part_order = ['year'];
        break;
    }

    $element['value']['#date_part_order'] = $date_part_order;

    // N'appliquer le year_range que s'il n'est pas vide
    if (!empty($year_range)) {
      $element['value']['#date_year_range'] = $year_range;
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);

    $element['date_order'] = [
      '#type' => 'select',
      '#title' => $this->t('Date part order'),
      '#default_value' => $this->getSetting('date_order'),
      '#options' => [
        'YM' => $this->t('Year/Month'),
        'MY' => $this->t('Month/Year'),
        'Y' => $this->t('Year'),
      ],
    ];

    $element['year_range'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Year range (optional)'),
      '#description' => $this->t('Leave empty for no restriction. Examples: 0:+10, 2000:2025, 2025:+5'),
      '#default_value' => $this->getSetting('year_range'),
      '#placeholder' => $this->t('No restriction'),
    ];

    $element['day_option'] = [
      '#type' => 'select',
      '#title' => $this->t('Day of month'),
      '#description' => $this->t('Choose which day of the month to use when only month/year are selected.'),
      '#default_value' => $this->getSetting('day_option'),
      '#options' => [
        'first' => $this->t('First day of month (1st)'),
        'last' => $this->t('Last day of month'),
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $date_order = $this->getSetting('date_order');
    $date_order_labels = [
      'YM' => $this->t('Year/Month'),
      'MY' => $this->t('Month/Year'),
      'Y' => $this->t('Year'),
    ];
    $summary[] = $this->t('Date order: @order', ['@order' => $date_order_labels[$date_order]]);

    $year_range = $this->getSetting('year_range');
    if (!empty($year_range)) {
      $summary[] = $this->t('Year range: @range', ['@range' => $year_range]);
    } else {
      $summary[] = $this->t('Year range: No restriction');
    }

    $day_option = $this->getSetting('day_option');
    $day_labels = [
      'first' => $this->t('First day'),
      'last' => $this->t('Last day'),
    ];
    $summary[] = $this->t('Day of month: @day', ['@day' => $day_labels[$day_option]]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Traitement des valeurs de formulaire
    foreach ($values as &$item) {
      if (!empty($item['value']) && $item['value'] instanceof DrupalDateTime) {
        // Appliquer l'option de jour
        $day_option = $this->getSetting('day_option');

        if ($day_option === 'last') {
          // Définir au dernier jour du mois
          $item['value']->setDate(
            $item['value']->format('Y'),
            $item['value']->format('n'),
            $item['value']->format('t') // 't' = nombre de jours dans le mois
          );
        } else {
          // Définir au premier jour du mois (par défaut)
          $item['value']->setDate(
            $item['value']->format('Y'),
            $item['value']->format('n'),
            1
          );
        }

        // Déterminer le format selon le type de champ
        $field_definition = $this->fieldDefinition;
        $datetime_type = $field_definition->getSetting('datetime_type');

        if ($datetime_type === 'date') {
          // Format date seulement
          $item['value'] = $item['value']->format('Y-m-d');
        } else {
          // Format datetime complet
          $item['value'] = $item['value']->format('Y-m-d\TH:i:s');
        }
      }
      elseif (empty($item['value'])) {
        $item['value'] = NULL;
      }
    }
    return $values;
  }

}
