<?php

namespace Drupal\month_year_range\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\datetime_range\Plugin\Field\FieldWidget\DateRangeDatelistWidget;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Plugin implementation of the 'month_year_range' widget.
 *
 * @FieldWidget(
 *   id = "month_year_range",
 *   label = @Translation("Month Year range"),
 *   field_types = {
 *     "daterange"
 *   }
 * )
 */
class MonthYearRangeWidget extends DateRangeDatelistWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'date_order' => 'YM',
        'year_range' => '',
        'day_option_start' => 'first',
        'day_option_end' => 'last',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $date_order = $this->getSetting('date_order');
    $year_range = $this->getSetting('year_range');

    // Set up the date part order array.
    switch ($date_order) {
      default:
      case 'YM':
        $date_part_order = ['year', 'month'];
        break;

      case 'MY':
        $date_part_order = ['month', 'year'];
        break;

      case 'Y':
        $date_part_order = ['year'];
        break;
    }

    $element['value'] = [
        '#type' => 'datelist',
        '#date_part_order' => $date_part_order,
      ] + $element['value'];

    $element['end_value'] = [
        '#type' => 'datelist',
        '#date_part_order' => $date_part_order,
        '#required' => FALSE,
      ] + $element['end_value'];

    // Appliquer le year_range si défini
    if (!empty($year_range)) {
      $element['value']['#date_year_range'] = $year_range;
      $element['end_value']['#date_year_range'] = $year_range;
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element = parent::settingsForm($form, $form_state);

    $element['date_order'] = [
      '#type' => 'select',
      '#title' => $this->t('Date part order'),
      '#default_value' => $this->getSetting('date_order'),
      '#options' => [
        'YM' => $this->t('Year/Month'),
        'MY' => $this->t('Month/Year'),
        'Y' => $this->t('Year'),
      ],
    ];

    $element['year_range'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Year range (optional)'),
      '#description' => $this->t('Leave empty for no restriction. Examples: 0:+10, 2000:2025, 2025:+5'),
      '#default_value' => $this->getSetting('year_range'),
      '#placeholder' => $this->t('No restriction'),
    ];

    $element['day_option_start'] = [
      '#type' => 'select',
      '#title' => $this->t('Start date - Day of month'),
      '#description' => $this->t('Choose which day of the month to use for the start date when only month/year are selected.'),
      '#default_value' => $this->getSetting('day_option_start'),
      '#options' => [
        'first' => $this->t('First day of month (1st)'),
        'last' => $this->t('Last day of month'),
      ],
    ];

    $element['day_option_end'] = [
      '#type' => 'select',
      '#title' => $this->t('End date - Day of month'),
      '#description' => $this->t('Choose which day of the month to use for the end date when only month/year are selected.'),
      '#default_value' => $this->getSetting('day_option_end'),
      '#options' => [
        'first' => $this->t('First day of month (1st)'),
        'last' => $this->t('Last day of month'),
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $date_order = $this->getSetting('date_order');
    $date_order_labels = [
      'YM' => $this->t('Year/Month'),
      'MY' => $this->t('Month/Year'),
      'Y' => $this->t('Year'),
    ];
    $summary[] = $this->t('Date order: @order', ['@order' => $date_order_labels[$date_order]]);

    $year_range = $this->getSetting('year_range');
    if (!empty($year_range)) {
      $summary[] = $this->t('Year range: @range', ['@range' => $year_range]);
    } else {
      $summary[] = $this->t('Year range: No restriction');
    }

    $day_option_start = $this->getSetting('day_option_start');
    $day_option_end = $this->getSetting('day_option_end');
    $day_labels = [
      'first' => $this->t('First day'),
      'last' => $this->t('Last day'),
    ];
    $summary[] = $this->t('Start date: @day', ['@day' => $day_labels[$day_option_start]]);
    $summary[] = $this->t('End date: @day', ['@day' => $day_labels[$day_option_end]]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    // Traitement des valeurs de formulaire
    foreach ($values as &$item) {
      // Traiter la date de début
      if (!empty($item['value']) && $item['value'] instanceof DrupalDateTime) {
        $this->applyDayOption($item['value'], $this->getSetting('day_option_start'));
        $item['value'] = $this->formatDate($item['value']);
      }

      // Traiter la date de fin
      if (!empty($item['end_value']) && $item['end_value'] instanceof DrupalDateTime) {
        $this->applyDayOption($item['end_value'], $this->getSetting('day_option_end'));
        $item['end_value'] = $this->formatDate($item['end_value']);
      }

      // Nettoyer les valeurs vides
      if (empty($item['value'])) {
        $item['value'] = NULL;
      }
      if (empty($item['end_value'])) {
        $item['end_value'] = NULL;
      }
    }
    return $values;
  }

  /**
   * Applique l'option de jour à une date.
   */
  private function applyDayOption(DrupalDateTime $date, $day_option) {
    if ($day_option === 'last') {
      // Définir au dernier jour du mois
      $date->setDate(
        $date->format('Y'),
        $date->format('n'),
        $date->format('t') // 't' = nombre de jours dans le mois
      );
    } else {
      // Définir au premier jour du mois (par défaut)
      $date->setDate(
        $date->format('Y'),
        $date->format('n'),
        1
      );
    }
  }

  /**
   * Formate la date selon le type de champ.
   */
  private function formatDate(DrupalDateTime $date) {
    // Déterminer le format selon le type de champ
    $field_definition = $this->fieldDefinition;
    $datetime_type = $field_definition->getSetting('datetime_type');

    if ($datetime_type === 'date') {
      // Format date seulement
      return $date->format('Y-m-d');
    } else {
      // Format datetime complet
      return $date->format('Y-m-d\TH:i:s');
    }
  }

}
