<?php

namespace Drupal\more_select_widgets\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsSelectWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'extendable_select' widget.
 *
 * PROTOTYPE - NOT WORKING YET.
 */
#[FieldWidget(
  id: 'extendable_select',
  label: new TranslatableMarkup('A select list with flags that can alter behavior.'),
  description: new TranslatableMarkup('Uses a plain string as the field type, but allows defining options with flags to modify behavior.'),
  field_types: ['none-yet'],
  multiple_values: TRUE,
)]

class ExtendableSelectWidget extends OptionsSelectWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'select_options' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $existing_options = $this->getSetting('select_options');

    $oelement['select_options'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Options for field'),
      '#description' => $this->t('One option per line of the format:
        key|label|flags; where "flags" are optional with possible values of "hide".
        You cannot delete existing entries - you must "hide" them.'),
      '#default_value' => $existing_options,
      '#rows' => 10,
      '#element_validate' => [[$this, 'settingsFormValidate']],
    ];
    return $oelement;
  }

  /**
   * Form callback to validate the options entered.
   */
  public function settingsFormValidate($element, FormStateInterface $form_state) {
    $new_value = $form_state->getValue($element['#parents']);
    if (!$new_value) {
      return;
    }
    $new_options = [];
    foreach (explode(PHP_EOL, $new_value) as $option_line) {
      if (empty(trim($option_line))) {
        continue;
      }
      $ovalues = explode('|', $option_line);
      if (count($ovalues) < 2) {
        $form_state->setError($element, $this->t('Each line must have at least a key and label separated by "|"'));
        return;
      }
      $option = ['key' => trim($ovalues[0]), 'name' => trim($ovalues[1])];
      if (count($ovalues) > 2) {
        $option['flags'] = trim($ovalues[2]);
      }
      $new_options[] = $option;
    }

    $new_keys = [];
    if ($new_options) {
      // Check for duplicate keys.
      foreach ($new_options as $option) {
        $new_keys[] = $option['key'];
      }
      if (count($new_keys) != count(array_unique($new_keys))) {
        $form_state->setError($element, $this->t("Duplicate keys not allowed"));
        return;
      }
    }

    $current_options = $this->convertToArray($this->getSetting('select_options'));
    if ($current_options) {
      $old_keys = [];
      foreach ($current_options as $current_option) {
        $old_keys[] = $current_option['key'];
      }
      if (!empty(array_diff($old_keys, $new_keys))) {
        $form_state->setError($element, "Keys cannot be deleted - you can hide a key from use by adding '|hide' to the end of line");
      }
    }
  }

  /**
   * Utility to take the textarea input and convert to array.
   */
  private function convertToArray($input) {
    $new_options = [];
    foreach (explode(PHP_EOL, $input) as $option_line) {
      if (empty($option_line)) {
        continue;
      }
      $ovalues = explode('|', $option_line);
      $flag_list = [];
      $flags = trim($ovalues[2] ?? '');
      if (!empty($flags)) {
        $flag_list = explode(',', $flags);
      }
      $new_options[] = [
        'key' => trim($ovalues[0]),
        'name' => trim($ovalues[1]),
        'flags' => $flag_list,
      ];
    }
    return $new_options;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    return ['select_options' => "Options: {$this->getSetting('select_options')}"];
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $options = $this->getOptions($items->getEntity());
    $element['#options'] = $options;
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  protected function getOptions(FieldableEntityInterface $entity) {
    if (!isset($this->options)) {
      $select_options = $this->convertToArray($this->getSetting('select_options'));
      $options = [];
      foreach ($select_options as $select_option) {
        if (in_array('hide', $select_option['flags'] ?? [])) {
          continue;
        }
        $options[$select_option['key']] = $select_option['name'];
      }

      // Add an empty option if the widget needs one.
      if ($empty_label = $this->getEmptyLabel()) {
        $options = ['_none' => $empty_label] + $options;
      }

      $module_handler = \Drupal::moduleHandler();
      $context = [
        'fieldDefinition' => $this->fieldDefinition,
        'entity' => $entity,
        'widget' => $this,
      ];
      $module_handler->alter('options_list', $options, $context);
      $this->options = $options;
    }
    return $this->options;
  }

}
