<?php

namespace Drupal\more_select_widgets\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsSelectWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'filterable_select' widget.
 */
#[FieldWidget(
    id: 'filterable_select',
    label: new TranslatableMarkup('Allowing filtering of select options.'),
    field_types: [
      'list_string',
    ],
    multiple_values: TRUE,
)]
class FilterableSelectWidget extends OptionsSelectWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'filtered_options' => [],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $filtered_options = $this->getFilteredOptions();
    // 'allowed_values' and 'allowed_values_function'
    $field_settings = $this->getFieldSettings();
    $allowed_values = $field_settings['allowed_values'] ?? [];
    $element['filtered_options'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Key'),
        $this->t('Value'),
        $this->t('Hide?'),
      ],
      '#attributes' => [
        'data-field-list-table' => TRUE,
        'class' => ['allowed-values-table'],
      ],
      '#attached' => [
        'library' => [
          'core/drupal.fieldListKeyboardNavigation',
          'field_ui/drupal.field_ui',
        ],
      ],
    ];
    $delta = 0;
    foreach ($allowed_values as $key => $value) {
      $element['filtered_options'][$delta] = [
        'key' => [
          '#type' => 'textfield',
          '#default_value' => $key,
          '#attributes' => ['readonly' => 'readonly'],
        ],
        'value' => [
          '#type' => 'textfield',
          '#default_value' => $value,
          '#attributes' => ['readonly' => 'readonly'],
        ],
        'hide' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Hide'),
          '#default_value' => $filtered_options[$key],
        ],
      ];
      $delta++;
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $settings = $this->getSetting('filtered_options');
    $total = count($settings);
    $nhidden = 0;
    foreach ($settings as $setting) {
      if ($setting['hide']) {
        $nhidden++;
      }
    }
    return ['select_options' => "Hidden options: {$nhidden} of {$total}"];
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $av = $this->getFieldSettings();
    // N.B. this is where the options_list_access hook is applied.
    $options = $this->getOptions($items->getEntity());

    $filtered_options = $this->getFilteredOptions();

    /* start with getOptions - which limits based on access etc.
     * then filter further based on hidden settings
     * The difficulty is that getOptions is just display values - not keys.
     */
    $displayed_options = [];
    foreach ($options as $key => $option) {
      $okey = array_search($option, $av['allowed_values']);
      if ($okey !== FALSE && array_key_exists($okey, $filtered_options) && $filtered_options[$okey] && !$this->isSelected($okey, $items)) {
        continue;
      }
      $displayed_options[$key] = $option;
    }
    $element['#options'] = $displayed_options;
    return $element;
  }

  /**
   * Return True if key is the currently selected value for the field.
   */
  private function isSelected($key, FieldItemListInterface $items) {
    foreach ($items as $item) {
      $value = $item->{$this->column};
      if ($value == $key) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Return an array of key: hide.
   */
  private function getFilteredOptions() {
    $filtered_options = $this->getSetting('filtered_options');
    $filtered = [];
    foreach ($filtered_options as $fo) {
      $filtered[$fo['key']] = (bool) $fo['hide'];
    }
    return $filtered;
  }

}
