<?php

namespace Drupal\more_select_widgets\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsSelectWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Plugin implementation of the 'taxonomy_select' widget.
 */
#[FieldWidget(
    id: 'taxonomy_select',
    label: new TranslatableMarkup('Build select list from taxonomy'),
    field_types: [
      'string',
    ],
    multiple_values: TRUE,
)]
class TaxonomySelectWidget extends OptionsSelectWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'vocabulary' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $vs = Vocabulary::loadMultiple();
    $voptions = [];
    foreach ($vs as $v) {
      $voptions[$v->id()] = $v->label();
    }
    $velement['vocabulary'] = [
      '#type' => 'radios',
      '#title' => $this->t('Choose which taxonomy vocabulary to select from'),
      '#options' => $voptions,
      '#default_value' => $this->getSetting('vocabulary'),
    ];
    return $velement;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    return ['vocabulary' => "Vocabulary: {$this->getSetting('vocabulary')}"];
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $options = $this->getOptions($items->getEntity());

    // We always want to allow something that is already there. This allows for example a term to be deleted, but still
    // be usable for existing records.
    foreach ($items as $item) {
      $value = $item->{$this->column};
      // Keep the value if it actually is in the list of options (needs to be
      // checked against the flat list).
      if (!isset($options[$value])) {
        $options[$value] = $value;
      }
    }

    $element['#options'] = $options;
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  protected function getOptions(FieldableEntityInterface $entity) {
    if (!isset($this->options)) {
      $query = \Drupal::entityQuery('taxonomy_term');
      $query->condition('vid', $this->settings['vocabulary']);
      if (!\Drupal::currentUser()->hasPermission('administer taxonomy')) {
        $query->condition('status', 1);
      }
      $tids = $query->accessCheck(TRUE)->execute();
      $terms = Term::loadMultiple($tids);
      $options = [];
      foreach ($terms as $term) {
        $name = $term->get('name')->getString();
        $options[$name] = $name;
      }

      // Add an empty option if the widget needs one.
      if ($empty_label = $this->getEmptyLabel()) {
        $options = ['_none' => $empty_label] + $options;
      }

      $module_handler = \Drupal::moduleHandler();
      $context = [
        'fieldDefinition' => $this->fieldDefinition,
        'entity' => $entity,
        'widget' => $this,
      ];
      $module_handler->alter('options_list', $options, $context);
      $this->options = $options;
    }
    return $this->options;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();
    $vid = $this->getSetting('vocabulary');
    if ($vid && $vocabulary = Vocabulary::load($vid)) {
      $dependencies['config'][] = $vocabulary->getConfigDependencyName();
    }
    return $dependencies;
  }

}
