<?php

namespace Drupal\ms_graph_api\Plugin\KeyInput;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Uuid\Uuid;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\key\KeyInterface;
use Drupal\key\Plugin\KeyInputBase;
use Drupal\key\Plugin\KeyProviderInterface;
use Drupal\ms_graph_api\Constants;

/**
 * Defines a key input for MS Graph API keys.
 *
 * This includes:
 *   - Tenant Domain.
 *   - Tenant ID.
 *   - Client ID.
 *   - Client Secret.
 *
 * @KeyInput(
 *   id = "ms_graph_api",
 *   label = @Translation("MS Graph API Key"),
 *   description = @Translation("Allows entry of the information required to authenticate with Microsoft Graph API.")
 * )
 *
 * @noinspection PhpUnused
 */
class GraphApiKeyInput extends KeyInputBase {

  /**
   * A regular expression that matches valid-looking domain names.
   */
  const VALID_DOMAIN_REGEX = '/^([0-9a-z-]+\.?)+\.[^\.]+$/';

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'key_value'     => '',
      'tenant_domain' => '',
      'tenant_id'     => '',
      'client_id'     => '',
      'client_secret' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form[Constants::KEY_VALUE_TENANT_DOMAIN] = [
      '#type'        => 'textfield',
      '#title'       => $this->t('Tenant Domain'),
      '#description' => $this->t('The primary domain of the Azure tenant.'),
      '#placeholder' => 'contoso.microsoft.com',
      '#required'    => TRUE,
    ];

    $form[Constants::KEY_VALUE_TENANT_ID] = [
      '#type'        => 'textfield',
      '#title'       => $this->t('Tenant ID'),
      '#description' => $this->t('The GUID/UUID for the Azure tenant.'),
      '#placeholder' => '123e4567-e89b-12d3-a456-426614174000',
      '#required'    => TRUE,
    ];

    $form[Constants::KEY_VALUE_CLIENT_ID] = [
      '#type'        => 'textfield',
      '#title'       => $this->t('Client ID'),
      '#description' => $this->t('The GUID/UUID for the graph API client (an app registration).'),
      '#placeholder' => '123e4567-e89b-12d3-a456-426614174000',
      '#required'    => TRUE,
    ];

    $form[Constants::KEY_VALUE_CLIENT_SECRET] = [
      '#type'        => 'textfield',
      '#title'       => $this->t('Client secret'),
      '#description' => $this->t('The secret key generated by Graph API.'),
      '#required'    => TRUE,
      '#sensitive'   => TRUE,
    ];

    $this->obscureAndPopulateDefaults($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   *
   * The key value is prepared for use by the input form by converting it from
   * JSON into an array of values.
   *
   * @return array
   *   The processed key value.
   */
  public function processExistingKeyValue($key_value): array {
    return Json::decode($key_value);
  }

  /**
   * {@inheritdoc}
   *
   * The key value is prepared for storage by first replacing any obscured
   * values with their stored copies, and then by converting the values from an
   * array into a JSON payload.
   */
  public function processSubmittedKeyValue(
                                        FormStateInterface $form_state): array {
    // This is the entire key edit form, not just the key input form.
    $key_edit_form = $form_state->getCompleteForm();

    assert(
      isset($key_edit_form['settings']['input_section']['key_input_settings'])
    );

    // This is the key input form, as constructed above in
    // buildConfigurationForm().
    $key_input_settings_form =&
      $key_edit_form['settings']['input_section']['key_input_settings'];

    $this->replaceObscuredFieldValues($key_input_settings_form, $form_state);

    $submitted_values = $form_state->getValues();
    $tenant_domain    = $submitted_values[Constants::KEY_VALUE_TENANT_DOMAIN];
    $tenant_id        = $submitted_values[Constants::KEY_VALUE_TENANT_ID];
    $client_id        = $submitted_values[Constants::KEY_VALUE_CLIENT_ID];

    if (!preg_match(self::VALID_DOMAIN_REGEX, $tenant_domain)) {
      $form_state->setError(
        $key_input_settings_form[Constants::KEY_VALUE_TENANT_DOMAIN],
        $this->t('Tenant Domain must be a valid second-level domain name or a sub-domain of a second-level domain.')
      );
    }

    if (!Uuid::isValid($tenant_id)) {
      $form_state->setError(
        $key_input_settings_form[Constants::KEY_VALUE_TENANT_ID],
        $this->t('Tenant ID must be a valid UUID.')
      );
    }

    if (!Uuid::isValid($client_id)) {
      $form_state->setError(
        $key_input_settings_form[Constants::KEY_VALUE_CLIENT_ID],
        $this->t('Client ID must be a valid UUID.')
      );
    }

    // Convert multiple form field values to serialized form.
    $encoded_values = Json::encode($submitted_values);

    $values_to_store = [
      'key_value' => $encoded_values,
    ];

    // Act as if the user had submitted the key fields as a JSON payload.
    $form_state->setValues($values_to_store);

    return parent::processSubmittedKeyValue($form_state);
  }

  /**
   * Gets the key provider for the key being edited by this input form.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the input form.
   *
   * @return \Drupal\key\Plugin\KeyProviderInterface
   *   The key provider for the key being edited.
   */
  protected function getKeyProvider(
                        FormStateInterface $form_state): KeyProviderInterface {
    $entity_form = $form_state->getFormObject();
    assert($entity_form instanceof EntityFormInterface);

    $key = $entity_form->getEntity();
    assert($key instanceof KeyInterface);

    return $key->getKeyProvider();
  }

  /**
   * Populates default values in the form, obscuring sensitive values.
   *
   * We have to do this ourselves because of separation-of-concerns design
   * issues in the Key module described in DDO-3168120. Basically, the Key
   * module wants to store the key data with the provider as a JSON payload, but
   * it handles obscuring of key values at the provider level. If we allowed Key
   * to handle obscuring the value, we'd end up with part of the JSON payload
   * being obscured, which would result in a payload we can't parse.
   *
   * @param array $form
   *   A reference to the key input form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the input form.
   */
  protected function obscureAndPopulateDefaults(
                                        array &$form,
                                        FormStateInterface $form_state): void {
    $key_provider = $this->getKeyProvider($form_state);

    $key_value_data = $form_state->get('key_value');
    $decoded_values = $key_value_data['current'] ?? [];

    foreach (Element::children($form) as $key) {
      $form_element =& $form[$key];

      $is_sensitive   = $form_element['#sensitive'] ?? FALSE;
      $original_value = $decoded_values[$key] ?? '';

      if ($is_sensitive) {
        $value =
          $this->obscureAuthenticationField($key_provider, $original_value);

        // Keep track of all variants of the value of this field in the form,
        // for introspection during form submission. This is necessary because
        // of limitations noted in DDO-3168120.
        $form_element['#original_value'] = $original_value;
        $form_element['#obscured_value'] = $value;
      }
      else {
        $value = $original_value;
      }

      $form_element['#default_value'] = $value;
    }
  }

  /**
   * Replaces any submitted values that match obscure values with stored values.
   *
   * We have to do this ourselves because of separation-of-concerns design
   * issues in the Key module described in DDO-3168120. Basically, the Key
   * module wants to store the key data with the provider as a JSON payload, but
   * it handles obscuring of key values at the provider level. If we allowed Key
   * to handle obscuring the value, we'd end up with part of the JSON payload
   * being obscured, which would result in a payload we can't parse.
   *
   * @param array $key_input_settings_form
   *   The form array of the key input form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the input form.
   */
  protected function replaceObscuredFieldValues(
                                              array $key_input_settings_form,
                                              FormStateInterface $form_state) {
    $submitted_values = $form_state->getValues();

    foreach ($submitted_values as $key => $value) {
      $obscured_value =
        $key_input_settings_form[$key]['#obscured_value'] ?? NULL;

      if (($obscured_value !== NULL) && ($value === $obscured_value)) {
        $original_value =
          $key_input_settings_form[$key]['#original_value'] ?? NULL;

        // Reset to the stored, original value so we don't overwrite the value
        // with the obscured version of the value.
        $form_state->setValue($key, $original_value);
      }
    }
  }

  /**
   * Obscures an authentication field value using the key provider.
   *
   * @param \Drupal\key\Plugin\KeyProviderInterface $key_provider
   *   The key provider that will be used to obscure the value.
   * @param string $value
   *   The value to obscure.
   *
   * @return string
   *   The obscured value.
   */
  protected function obscureAuthenticationField(
                                            KeyProviderInterface $key_provider,
                                            string $value): string {
    $obscure_options = [
      'key_type_group' => 'authentication',
    ];

    // As noted in DDO-3168120, obscureKeyValue() is a static method defined on
    // the interface... #FML.
    return $key_provider->obscureKeyValue($value, $obscure_options);
  }

}
