<?php

namespace Drupal\multi_domain_login\Controller;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use function GuzzleHttp\Psr7\str;

/**
 * Returns responses for Login routes.
 *
 * Login multiple domains:
 *
 * Example:
 * domain A
 * domain B
 * domain C
 *
 * Step 1 - login current domain
 *
 * - Do normal drupal login at domain A: http://www.a.com/user/login
 * - hook_user_login is called after user has logged in.
 *   generate a random hash for domain B (hash-b) and C (hash-c).
 *   store in 'key_value' with hash as key and current uid as value.
 *   redirect to: http://www.a.com/user/login/domain
 *
 * Step 2 - http://www.a.com/user/login/domain page
 * - does a call in page to each domain from iframe
 *     http://www.b.com/user/login/domain/uid/timestamp/hash-b
 *     http://www.c.com/user/login/domain/uid/timestamp/hash-c
 * - after success on all domains redirect to homepage of original domain (event bubble):
 *     http://www.a.com
 * - if failure to login, logout
 *     http://www.a.com/logout
 *
 * Step 3 - http://www.x.com/user/login/domain/hash-x
 * - lookup hash-x in 'key_value' storage to retrieve uid.
 * - login user
 * - remove hash-x from storage
 * - return ok
 *
 */
class MultiDomainLoginController extends ControllerBase implements ContainerInjectionInterface {

  /**
   * A logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a UserController object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   *
   */
  public function __construct(LoggerInterface $logger) {
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.factory')->get('multi_domain_login')
    );
  }

  // Entry point after the user_login hook, starts the redirect flow
  // through all the domains by redirecting to the first domain login route.
  // We keep hold of the initial domain making the request (referrer) and store
  // it in the url as a crc32 encoded value to prevent any possible url encoding
  // issues in the url.
  public function domain(Request $request) {
    \Drupal::service('page_cache_kill_switch')->trigger();

    $referrer = $this->getRequestDomain($request);
    $url = $this->loginUrl($request, $referrer, TRUE);

    $response  = new TrustedRedirectResponse($url, 303);
    $response->addCacheableDependency((new \Drupal\Core\Cache\CacheableMetadata())
      ->setCacheMaxAge(0)
    );

    return  $response;
  }

  // A domain specific login. Incoming is the referrer and the domain on which we want to login.
  //
  public function login($referrer, $uid, $timestamp, $hash, Request $request) {
    \Drupal::service('page_cache_kill_switch')->trigger();

    // Do the login on the current domain.
    $this->doLogin($uid, $timestamp, $hash);

    // Get the next login url (or the initial home page).
    $url = $this->loginUrl($request, $referrer, FALSE);

    $response = new TrustedRedirectResponse($url, 303);
    $response->addCacheableDependency((new \Drupal\Core\Cache\CacheableMetadata())
      ->setCacheMaxAge(0)
    );

    return $response;
  }

  // Helper to generate a domain specific login url.
  protected function loginUrl($request, $referrer, $skip_referrer_check) {

    // Get the current domain.
    $current = $this->getRequestDomain($request);

    $timestamp = \Drupal::time()->getRequestTime();
    $user = \Drupal::currentUser();

    /** @var \Drupal\user\UserInterface $account */
    $account = $this->entityTypeManager()->getStorage('user')->load($user->id());

    // Get the domains we need to redirect to.
    $domains = $this->getDomains();

    // Look for the current domain
    while (crc32(current($domains)) != $current) {
      next($domains);
    }
    // Take the next in the array, or loop back to the first.
    $domain = next($domains) ?: reset($domains);

    if (!$skip_referrer_check && crc32($domain) == $referrer) {
      // Generate a language independent path to the domain.

      $urlToRedirectONSuccess = \Drupal::config('multi_domain_login.settings')->get('redirect_success');
      if (empty($urlToRedirectONSuccess)) {
        $url = Url::fromRoute('<front>');
      } else {
        $url = Url::fromUserInput($urlToRedirectONSuccess);
      }

      $url = $url->setAbsolute()
        ->toString(true)
        ->getGeneratedUrl();

      // Url will be in the current domain, replace that with the requested domain.
      $url = str_replace($domains, $domain, $url);
    }
    else {
      // Generate the login url for the next domain.

      // Generate the hash for this user.
      $hash = $this->hash($account, $timestamp);

      // Generate a language independent path to login to the domain.
      $url = Url::fromRoute('multi_domain_login.login', [
        'referrer' => $referrer,
        'uid' => $account->id(),
        'timestamp' => $timestamp,
        'hash' => $hash
      ])->setAbsolute()
        ->toString(true)
        ->getGeneratedUrl();

      // Url will be in the current domain, replace that with the requested domain.
      $url = str_replace($domains, $domain, $url);
    }

    // Let other modules alter the url if needed.
    \Drupal::moduleHandler()->alter('multi_domain_login_url', $url, $domain);

    return $url;
  }

  protected function getRequestDomain(Request $request) {

    // Fallback to the current host if not found in the domains list.
    $domain = $request->getSchemeAndHttpHost();

    // Get the domains we need to redirect to.
    $domains = $this->getDomains();

    // Determine the domain based on the current request.
    foreach ($domains as $aDomain) {
      if (str_starts_with($request->getUri(), $aDomain)) {
        $domain = $aDomain;
        break;
      }
    }

    // Let other modules alter the current domain if needed.
    \Drupal::moduleHandler()->alter('multi_domain_login_domain', $domain, $domains);

    return crc32($domain);
  }



  /**
   * Helper to get the domains we need to login to.
   *
   * @return array|mixed|null
   */
  protected function getDomains() {
    // Get the domains we need to redirect to.
    $domains = $this->config('multi_domain_login.settings')->get('domains');
    \Drupal::moduleHandler()->alter('multi_domain_login_domains', $domains);
    return $domains;
  }

  protected function doLogin($uid, $timestamp, $hash) {

    $enableExtraLogging = $this->config('multi_domain_login.settings')->get('enable_extra_logging');
    $currentHost = \Drupal::request()->getHost();

    if ($this->timeout($timestamp)) {
      $status = 403;
      $this->logger->critical('Login attempt expired @domain', ['@domain' => $currentHost]);
    }
    else {
      /** @var \Drupal\user\UserInterface $user */
      $user = $this->entityTypeManager()->getStorage('user')->load($uid);

      // Verify that the user exists and is active.
      if ($user === NULL || !$user->isActive()) {
        // Blocked or invalid user ID, so deny access. The parameters will be in
        // the watchdog's URL for the administrator to check.
        $status = 403;
        $this->logger->warning('User @uid no longer active or found @domain', ['@uid' => $uid, '@domain' => $currentHost]);
      }
      else {

        $current_user = \Drupal::currentUser();
        $force_logout = $this->config('multi_domain_login.settings')->get('force_logout');

        if ($current_user->isAuthenticated() && !$force_logout) {
          $this->logger->info('User @uid already logged in @domain', ['@uid' => $uid, '@domain' => $currentHost]);
          $status = 200;
        }
        else if ($current_user->isAuthenticated() && $force_logout) {
          user_logout();
          if ($enableExtraLogging) {
            $this->logger->debug('User logout @domain', ['@domain' => $currentHost]);
          }
        }

        // Log in with our requested user.
        if ($current_user->isAnonymous()) {
          if (hash_equals($hash, $this->hash($user, $timestamp))) {
            user_login_finalize($user);
            $status = 200;
            if ($enableExtraLogging) {
              $this->logger->debug('Login finalize: 200 @domain', ['@domain' => $currentHost]);
            }
          }
          else {
            $status = 403;
            $this->logger->critical('Invalid hash used in login attempt @domain', ['@domain' => $currentHost]);
          }
        }
      }
    }

    return $status;
  }

  protected function hash(UserInterface $account, $timestamp) {
    $data = $timestamp;
    $data .= $account->id();
    $data .= $account->getEmail();
    return Crypt::hmacBase64($data, Settings::getHashSalt() . $account->getPassword());
  }

  protected function timeout($timestamp) {
    // The current user is not logged in, so check the parameters.
    $current = \Drupal::time()->getRequestTime();

    // Time out, in seconds, until login URL expires.
    $timeout = $this->config('multi_domain_login.settings')->get('timeout');

    return ($current - $timestamp > $timeout);
  }
}
