<?php

declare(strict_types=1);

namespace Drupal\Tests\multi_menu_ui\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Tests the multi_menu_link field type.
 *
 * @group multi_menu_ui
 */
class MultiMenuLinkFieldTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'multi_menu_ui',
    'node',
    'user',
    'system',
    'field',
    'link',
    'menu_link_content',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('menu_link_content');
    $this->installSchema('system', ['sequences']);
    $this->installSchema('node', ['node_access']);

    // Create test content type.
    $node_type = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $node_type->save();
  }

  /**
   * Tests field creation.
   */
  public function testFieldCreation(): void {
    // Create field storage.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
      'cardinality' => -1,
    ]);
    $field_storage->save();

    // Create field instance.
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
      'label' => 'Multi Menu Link',
    ]);
    $field->save();

    $this->assertInstanceOf(FieldConfig::class, $field);
    $this->assertEquals('multi_menu_link', $field->getType());
  }

  /**
   * Tests field schema.
   */
  public function testFieldSchema(): void {
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
    ]);
    $field_storage->save();

    $schema = $field_storage->getSchema();

    $this->assertArrayHasKey('columns', $schema);
    $this->assertArrayHasKey('menu_name', $schema['columns']);
    $this->assertArrayHasKey('parent', $schema['columns']);
    $this->assertArrayHasKey('weight', $schema['columns']);
    $this->assertArrayHasKey('title', $schema['columns']);
    $this->assertArrayHasKey('description', $schema['columns']);
    $this->assertArrayHasKey('enabled', $schema['columns']);
    $this->assertArrayHasKey('mlc_uuid', $schema['columns']);
  }

  /**
   * Tests isEmpty() behavior.
   */
  public function testFieldIsEmpty(): void {
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
    ]);
    $field->save();

    $node = $this->container->get('entity_type.manager')
      ->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => 'Test Node',
      ]);

    // Empty field should be empty.
    $this->assertTrue($node->get('field_multi_menu')->isEmpty());

    // Field with only menu_name should not be empty.
    $node->set('field_multi_menu', [
      'menu_name' => 'main',
      'parent' => '',
      'weight' => 0,
      'title' => '',
      'description' => '',
      'enabled' => 1,
    ]);
    $this->assertFalse($node->get('field_multi_menu')->isEmpty());

    // Field with empty menu_name should be empty.
    $node->set('field_multi_menu', [
      'menu_name' => '',
      'parent' => '',
      'weight' => 0,
      'title' => 'Test',
      'description' => '',
      'enabled' => 1,
    ]);
    $this->assertTrue($node->get('field_multi_menu')->isEmpty());
  }

  /**
   * Tests multiple field values.
   */
  public function testMultipleValues(): void {
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
      'cardinality' => -1,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
    ]);
    $field->save();

    $node = $this->container->get('entity_type.manager')
      ->getStorage('node')
      ->create([
        'type' => 'page',
        'title' => 'Test Node',
        'field_multi_menu' => [
          [
            'menu_name' => 'main',
            'parent' => '',
            'weight' => 0,
            'title' => 'Link 1',
            'description' => 'Description 1',
            'enabled' => 1,
          ],
          [
            'menu_name' => 'footer',
            'parent' => '',
            'weight' => 5,
            'title' => 'Link 2',
            'description' => 'Description 2',
            'enabled' => 1,
          ],
        ],
      ]);

    $this->assertEquals(2, $node->get('field_multi_menu')->count());
    $this->assertEquals('main', $node->get('field_multi_menu')->get(0)->menu_name);
    $this->assertEquals('footer', $node->get('field_multi_menu')->get(1)->menu_name);
  }

}
