<?php

declare(strict_types=1);

namespace Drupal\multi_menu_ui\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'additional_menu_link' field type.
 *
 * Stores ONLY additional menu links (not the primary one managed by core menu_ui).
 */
#[FieldType(
  id: 'additional_menu_link',
  label: new TranslatableMarkup('Additional Menu Link'),
  description: new TranslatableMarkup('Stores additional menu link definitions (the primary link is managed by core menu_ui).'),
  default_widget: 'string_textfield',
  default_formatter: 'string',
  cardinality: FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
)]
class AdditionalMenuLinkItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties['menu_name'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Menu name'))
      ->setRequired(TRUE);

    $properties['parent'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Parent menu link'))
      ->setDescription(new TranslatableMarkup('The plugin ID of the parent menu link.'));

    $properties['weight'] = DataDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Weight'))
      ->setDescription(new TranslatableMarkup('Link weight among links in the same menu at the same depth.'))
      ->setRequired(TRUE);

    $properties['title'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Menu link title'))
      ->setDescription(new TranslatableMarkup('The text to be used for this link in the menu. Leave empty to use the node title.'));

    $properties['description'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Description'))
      ->setDescription(new TranslatableMarkup('Shown when hovering over the menu link.'));

    $properties['enabled'] = DataDefinition::create('boolean')
      ->setLabel(new TranslatableMarkup('Enabled'))
      ->setDescription(new TranslatableMarkup('A flag for whether the link should be enabled in menus.'))
      ->setRequired(TRUE);

    $properties['mlc_uuid'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Menu Link Content UUID'))
      ->setDescription(new TranslatableMarkup('The UUID of the created menu_link_content entity.'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'menu_name' => [
          'description' => 'The menu name.',
          'type' => 'varchar',
          'length' => 32,
          'not null' => TRUE,
        ],
        'parent' => [
          'description' => 'The plugin ID of the parent menu link.',
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
        'weight' => [
          'description' => 'Weight among links in the same menu at the same depth.',
          'type' => 'int',
          'not null' => TRUE,
          'default' => 0,
        ],
        'title' => [
          'description' => 'The menu link title.',
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
        'description' => [
          'description' => 'The description shown when hovering over the menu link.',
          'type' => 'text',
          'size' => 'medium',
          'not null' => FALSE,
        ],
        'enabled' => [
          'description' => 'Whether the link should be enabled in menus.',
          'type' => 'int',
          'size' => 'tiny',
          'not null' => TRUE,
          'default' => 1,
        ],
        'mlc_uuid' => [
          'description' => 'The UUID of the created menu_link_content entity.',
          'type' => 'varchar',
          'length' => 128,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'menu_name' => ['menu_name'],
        'mlc_uuid' => ['mlc_uuid'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $menu_name = $this->get('menu_name')->getValue();
    return $menu_name === NULL || $menu_name === '';
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition): array {
    return [
      'menu_name' => 'footer',
      'parent' => '',
      'weight' => 0,
      'title' => 'Sample Additional Menu Link',
      'description' => 'This is an additional menu link.',
      'enabled' => 1,
      'mlc_uuid' => '',
    ];
  }

}
