<?php

declare(strict_types=1);

namespace Drupal\multi_menu_ui\Service;

use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\node\NodeInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for synchronizing additional menu links (excluding the primary one).
 *
 * The primary menu link is always managed by core menu_ui.
 * This service only handles additional menu links.
 */
class AdditionalMenuLinkSyncService {

  /**
   * Constructs an AdditionalMenuLinkSyncService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entityRepository
   *   The entity repository.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LoggerInterface $logger,
    private readonly EntityRepositoryInterface $entityRepository,
  ) {}

  /**
   * Syncs additional menu links for a node (excludes primary link).
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  public function syncAdditionalMenuLinks(NodeInterface $node): void {
    // Only sync for default translation to avoid duplicates.
    if (!$node->isDefaultTranslation()) {
      return;
    }

    $field_name = $this->findAdditionalMenuLinksField($node);
    if (!$field_name) {
      return;
    }

    // If field is empty, delete all additional links.
    if ($node->get($field_name)->isEmpty()) {
      $this->deleteAllAdditionalMenuLinks($node);
      return;
    }

    // Get the PRIMARY menu link UUID (managed by core menu_ui).
    // We must NEVER touch this one!
    $primary_uuid = $this->getPrimaryMenuLinkUuid($node);

    $field_items = $node->get($field_name)->getValue();
    $processed_uuids = [];

    // Step 1: Create or update additional menu links.
    foreach ($field_items as $delta => &$item) {
      if (empty($item['menu_name'])) {
        continue;
      }

      $uuid = $item['mlc_uuid'] ?? '';

      if ($uuid && $this->loadMenuLinkContentByUuid($uuid)) {
        // Update existing menu link.
        try {
          $this->updateMenuLink($uuid, $node, $item);
          $processed_uuids[] = $uuid;
        }
        catch (\Exception $e) {
          $this->logger->error('Failed to update additional menu link @uuid: @message', [
            '@uuid' => $uuid,
            '@message' => $e->getMessage(),
          ]);
        }
      }
      else {
        // Create new menu link.
        try {
          $new_uuid = $this->createMenuLink($node, $item);
          $item['mlc_uuid'] = $new_uuid;
          $processed_uuids[] = $new_uuid;
        }
        catch (\Exception $e) {
          $this->logger->error('Failed to create additional menu link for node @nid: @message', [
            '@nid' => $node->id(),
            '@message' => $e->getMessage(),
          ]);
        }
      }
    }

    // Update field with new UUIDs.
    $node->get($field_name)->setValue($field_items);

    // Step 2: Delete orphaned additional menu links (EXCLUDE primary!).
    $this->deleteOrphanedAdditionalLinks($node, $processed_uuids, $primary_uuid);
  }

  /**
   * Deletes all additional menu links for a node (excludes primary).
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  public function deleteAllAdditionalMenuLinks(NodeInterface $node): void {
    $primary_uuid = $this->getPrimaryMenuLinkUuid($node);
    $this->deleteOrphanedAdditionalLinks($node, [], $primary_uuid);
  }

  /**
   * Gets the UUID of the primary menu link (managed by core menu_ui).
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   *
   * @return string|null
   *   The UUID of the primary menu link, or NULL if none exists.
   */
  private function getPrimaryMenuLinkUuid(NodeInterface $node): ?string {
    // The primary link is the FIRST menu_link_content created for this node.
    // Core menu_ui creates this one.
    $storage = $this->entityTypeManager->getStorage('menu_link_content');

    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id())
      ->sort('id', 'ASC')  // First created = primary.
      ->range(0, 1);

    $ids = $query->execute();

    if (empty($ids)) {
      return NULL;
    }

    /** @var \Drupal\menu_link_content\Entity\MenuLinkContent $menu_link */
    $menu_link = $storage->load(reset($ids));
    return $menu_link ? $menu_link->uuid() : NULL;
  }

  /**
   * Finds the additional_menu_links field on the node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   *
   * @return string|null
   *   The field name or NULL if not found.
   */
  private function findAdditionalMenuLinksField(NodeInterface $node): ?string {
    $fields = $node->getFieldDefinitions();

    foreach ($fields as $field_name => $field_definition) {
      if ($field_definition->getType() === 'additional_menu_link') {
        return $field_name;
      }
    }

    return NULL;
  }

  /**
   * Creates a new menu_link_content entity.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param array $fieldItemData
   *   The field item data.
   *
   * @return string
   *   The UUID of the created menu link.
   *
   * @throws \Exception
   *   If the menu link cannot be created.
   */
  private function createMenuLink(NodeInterface $node, array $fieldItemData): string {
    $data = $this->prepareMenuLinkData($node, $fieldItemData);

    $menu_link = MenuLinkContent::create($data);
    $menu_link->save();

    $this->logger->info('Created additional menu link @uuid for node @nid in menu @menu', [
      '@uuid' => $menu_link->uuid(),
      '@nid' => $node->id(),
      '@menu' => $fieldItemData['menu_name'],
    ]);

    return $menu_link->uuid();
  }

  /**
   * Updates an existing menu_link_content entity.
   *
   * @param string $uuid
   *   The UUID of the menu link.
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param array $fieldItemData
   *   The field item data.
   *
   * @throws \Exception
   *   If the menu link cannot be updated.
   */
  private function updateMenuLink(string $uuid, NodeInterface $node, array $fieldItemData): void {
    $menu_link = $this->loadMenuLinkContentByUuid($uuid);
    if (!$menu_link) {
      $this->logger->warning('Additional menu link @uuid not found for update', ['@uuid' => $uuid]);
      throw new \Exception("Menu link {$uuid} not found");
    }

    $data = $this->prepareMenuLinkData($node, $fieldItemData);

    foreach ($data as $key => $value) {
      $menu_link->set($key, $value);
    }

    $menu_link->save();

    $this->logger->info('Updated additional menu link @uuid for node @nid', [
      '@uuid' => $uuid,
      '@nid' => $node->id(),
    ]);
  }

  /**
   * Deletes orphaned additional menu links (EXCLUDES the primary link!).
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param array $keep_uuids
   *   UUIDs of additional links to keep.
   * @param string|null $primary_uuid
   *   UUID of the primary menu link to NEVER delete.
   */
  private function deleteOrphanedAdditionalLinks(NodeInterface $node, array $keep_uuids, ?string $primary_uuid): void {
    // Find all menu links pointing to this node.
    $storage = $this->entityTypeManager->getStorage('menu_link_content');

    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());

    $mlc_ids = $query->execute();

    if (empty($mlc_ids)) {
      return;
    }

    /** @var \Drupal\menu_link_content\Entity\MenuLinkContent[] $menu_links */
    $menu_links = $storage->loadMultiple($mlc_ids);

    foreach ($menu_links as $menu_link) {
      $uuid = $menu_link->uuid();

      // CRITICAL: NEVER delete the primary link (managed by core menu_ui)!
      if ($uuid === $primary_uuid) {
        continue;
      }

      // Delete if not in keep list (it's an orphaned additional link).
      if (!in_array($uuid, $keep_uuids, TRUE)) {
        $this->logger->info('Deleting orphaned additional menu link @uuid for node @nid', [
          '@uuid' => $uuid,
          '@nid' => $node->id(),
        ]);
        $menu_link->delete();
      }
    }
  }

  /**
   * Loads a menu_link_content entity by UUID.
   *
   * @param string $uuid
   *   The UUID.
   *
   * @return \Drupal\menu_link_content\Entity\MenuLinkContent|null
   *   The menu link content entity or NULL.
   */
  private function loadMenuLinkContentByUuid(string $uuid): ?MenuLinkContent {
    try {
      $entity = $this->entityRepository->loadEntityByUuid('menu_link_content', $uuid);
      return $entity instanceof MenuLinkContent ? $entity : NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Prepares menu link data array.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param array $fieldItemData
   *   The field item data.
   *
   * @return array
   *   The menu link data array.
   *
   * @throws \InvalidArgumentException
   *   If the menu does not exist.
   */
  private function prepareMenuLinkData(NodeInterface $node, array $fieldItemData): array {
    // Validate menu exists.
    $menu_storage = $this->entityTypeManager->getStorage('menu');
    if (!$menu_storage->load($fieldItemData['menu_name'])) {
      throw new \InvalidArgumentException("Menu '{$fieldItemData['menu_name']}' does not exist.");
    }

    // Use field title or fallback to node title.
    $title = !empty($fieldItemData['title']) ? $fieldItemData['title'] : $node->getTitle();

    return [
      'title' => $title,
      'link' => ['uri' => 'entity:node/' . $node->id()],
      'menu_name' => $fieldItemData['menu_name'],
      'parent' => $fieldItemData['parent'] ?? '',
      'weight' => (int) ($fieldItemData['weight'] ?? 0),
      'description' => $fieldItemData['description'] ?? '',
      'enabled' => (bool) ($fieldItemData['enabled'] ?? TRUE),
      'langcode' => $node->language()->getId(),
    ];
  }

}
