<?php

declare(strict_types=1);

namespace Drupal\Tests\multi_menu_ui\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\system\Entity\Menu;

/**
 * Tests full CRUD operations for multi menu links.
 *
 * @group multi_menu_ui
 */
class MultiMenuLinkCrudTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'multi_menu_ui',
    'node',
    'user',
    'system',
    'field',
    'link',
    'menu_link_content',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('menu_link_content');
    $this->installSchema('system', ['sequences']);
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['system']);

    // Create test content type.
    $node_type = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $node_type->save();

    // Create field storage and field.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
      'cardinality' => -1,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
    ]);
    $field->save();

    // Create test menus.
    Menu::create([
      'id' => 'test-menu',
      'label' => 'Test Menu',
    ])->save();

    Menu::create([
      'id' => 'footer',
      'label' => 'Footer',
    ])->save();
  }

  /**
   * Tests node with multiple menu links.
   */
  public function testNodeWithMultipleMenuLinks(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Link 1',
          'description' => 'Description 1',
          'enabled' => 1,
        ],
        [
          'menu_name' => 'test-menu',
          'parent' => '',
          'weight' => 5,
          'title' => 'Link 2',
          'description' => 'Description 2',
          'enabled' => 1,
        ],
        [
          'menu_name' => 'footer',
          'parent' => '',
          'weight' => 10,
          'title' => 'Link 3',
          'description' => 'Description 3',
          'enabled' => 0,
        ],
      ],
    ]);
    $node->save();

    // Verify 3 menu links were created.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();

    $this->assertCount(3, $mlc_ids);

    // Load and verify each menu link.
    $menu_links = $storage->loadMultiple($mlc_ids);
    $titles = [];
    foreach ($menu_links as $menu_link) {
      $titles[] = $menu_link->getTitle();
    }

    $this->assertContains('Link 1', $titles);
    $this->assertContains('Link 2', $titles);
    $this->assertContains('Link 3', $titles);
  }

  /**
   * Tests deleting a node deletes all menu links.
   */
  public function testDeleteNode(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Link 1',
          'description' => '',
          'enabled' => 1,
        ],
        [
          'menu_name' => 'test-menu',
          'parent' => '',
          'weight' => 0,
          'title' => 'Link 2',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();
    $node_id = $node->id();

    // Verify menu links exist.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node_id);
    $mlc_ids = $query->execute();
    $this->assertCount(2, $mlc_ids);

    // Delete the node.
    $node->delete();

    // Verify menu links were deleted.
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node_id);
    $mlc_ids = $query->execute();
    $this->assertCount(0, $mlc_ids);
  }

  /**
   * Tests updating menu link title.
   */
  public function testUpdateMenuLinkTitle(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Original Title',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Update the title.
    $field_values = $node->get('field_multi_menu')->getValue();
    $field_values[0]['title'] = 'Updated Title';
    $node->set('field_multi_menu', $field_values);
    $node->save();

    // Verify title was updated.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $menu_link = $storage->load(reset($mlc_ids));

    $this->assertEquals('Updated Title', $menu_link->getTitle());
  }

  /**
   * Tests disabling menu link.
   */
  public function testDisableMenuLink(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Test Link',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Disable the menu link.
    $field_values = $node->get('field_multi_menu')->getValue();
    $field_values[0]['enabled'] = 0;
    $node->set('field_multi_menu', $field_values);
    $node->save();

    // Verify menu link is disabled.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $menu_link = $storage->load(reset($mlc_ids));

    $this->assertFalse($menu_link->isEnabled());
  }

  /**
   * Tests empty title uses node title.
   */
  public function testEmptyTitleUsesNodeTitle(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Node Title',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => '',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Verify menu link uses node title.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $menu_link = $storage->load(reset($mlc_ids));

    $this->assertEquals('Node Title', $menu_link->getTitle());
  }

}
