<?php

declare(strict_types=1);

namespace Drupal\Tests\multi_menu_ui\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\system\Entity\Menu;

/**
 * Tests the multi menu link sync service.
 *
 * @group multi_menu_ui
 */
class MultiMenuLinkSyncTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'multi_menu_ui',
    'node',
    'user',
    'system',
    'field',
    'link',
    'menu_link_content',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('menu_link_content');
    $this->installSchema('system', ['sequences']);
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['system']);

    // Create test content type.
    $node_type = NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ]);
    $node_type->save();

    // Create field storage and field.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_multi_menu',
      'entity_type' => 'node',
      'type' => 'multi_menu_link',
      'cardinality' => -1,
    ]);
    $field_storage->save();

    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'page',
    ]);
    $field->save();

    // Create test menu.
    Menu::create([
      'id' => 'test-menu',
      'label' => 'Test Menu',
    ])->save();
  }

  /**
   * Tests creating menu links on node save.
   */
  public function testCreateMenuLinksOnNodeSave(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Main Link',
          'description' => 'Main link description',
          'enabled' => 1,
        ],
        [
          'menu_name' => 'test-menu',
          'parent' => '',
          'weight' => 5,
          'title' => 'Test Link',
          'description' => 'Test link description',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Verify menu links were created.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();

    $this->assertCount(2, $mlc_ids);

    // Verify UUIDs were stored in field.
    $field_values = $node->get('field_multi_menu')->getValue();
    $this->assertNotEmpty($field_values[0]['mlc_uuid']);
    $this->assertNotEmpty($field_values[1]['mlc_uuid']);
  }

  /**
   * Tests updating existing menu links.
   */
  public function testUpdateMenuLinksOnNodeUpdate(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Original Title',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Get the UUID.
    $field_values = $node->get('field_multi_menu')->getValue();
    $uuid = $field_values[0]['mlc_uuid'];
    $this->assertNotEmpty($uuid);

    // Update the menu link.
    $node->set('field_multi_menu', [
      [
        'menu_name' => 'main',
        'parent' => '',
        'weight' => 10,
        'title' => 'Updated Title',
        'description' => 'Updated description',
        'enabled' => 1,
        'mlc_uuid' => $uuid,
      ],
    ]);
    $node->save();

    // Verify menu link was updated (not recreated).
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();

    $this->assertCount(1, $mlc_ids);

    // Load the menu link and verify changes.
    $menu_link = $storage->loadByProperties(['uuid' => $uuid]);
    $menu_link = reset($menu_link);
    $this->assertEquals('Updated Title', $menu_link->getTitle());
    $this->assertEquals(10, $menu_link->getWeight());
  }

  /**
   * Tests deleting orphaned menu links.
   */
  public function testDeleteOrphanedMenuLinks(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Link 1',
          'description' => '',
          'enabled' => 1,
        ],
        [
          'menu_name' => 'test-menu',
          'parent' => '',
          'weight' => 0,
          'title' => 'Link 2',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Verify 2 menu links were created.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $this->assertCount(2, $mlc_ids);

    // Remove one menu link from field.
    $field_values = $node->get('field_multi_menu')->getValue();
    $node->set('field_multi_menu', [$field_values[0]]);
    $node->save();

    // Verify only 1 menu link remains.
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $this->assertCount(1, $mlc_ids);
  }

  /**
   * Tests UUID tracking.
   */
  public function testMenuLinkUuidTracking(): void {
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Node',
      'field_multi_menu' => [
        [
          'menu_name' => 'main',
          'parent' => '',
          'weight' => 0,
          'title' => 'Test Link',
          'description' => '',
          'enabled' => 1,
        ],
      ],
    ]);
    $node->save();

    // Get the stored UUID.
    $field_values = $node->get('field_multi_menu')->getValue();
    $stored_uuid = $field_values[0]['mlc_uuid'];
    $this->assertNotEmpty($stored_uuid);

    // Load the menu link and verify UUIDs match.
    $storage = $this->container->get('entity_type.manager')->getStorage('menu_link_content');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('link.uri', 'entity:node/' . $node->id());
    $mlc_ids = $query->execute();
    $menu_link = $storage->load(reset($mlc_ids));

    $this->assertEquals($stored_uuid, $menu_link->uuid());
  }

}
