<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Cache\CacheableRedirectResponse;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailInterface;
use Drupal\user\UserInterface;

/**
 * Controller for multiple email confirmation.
 */
class EmailController implements ContainerInjectionInterface {

  use AutowireTrait;
  use StringTranslationTrait;

  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EmailConfirmerInterface $emailConfirmer,
    protected readonly MessengerInterface $messenger,
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * Validates the path parameters.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\multiple_email\EmailInterface $multiple_email
   *   The email.
   * @param string $code
   *   The hashed code.
   */
  protected function validatePathParameters(UserInterface $user, EmailInterface $multiple_email, string $code): bool {
    return hash_equals(Crypt::hmacBase64($code, Settings::getHashSalt() . $user->getPassword()), $code);
  }

  /**
   * Confirms the email address.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   The multiple_email entity cannot be saved.
   */
  public function confirmUser(UserInterface $user, EmailInterface $email, string $code): CacheableRedirectResponse {
    $attempts = $email->getAttempts();
    $confirmation_settings = $this->configFactory
      ->get('multiple_email.settings')
      ->get('confirm');
    $url = Url::fromRoute(
      'multiple_email.manage',
      ['user' => $user->id()],
      ['absolute' => TRUE],
    );

    if ($attempts < $confirmation_settings['attempts']) {
      if ($this->validatePathParameters($user, $email, $code)) {
        $email->setAttempts($attempts + 1)
          ->save();

        $this->messenger->addError($this->t('The confirmation code was incorrect.'));
      }
      else {
        $email->setConfirmed()
          ->save();
        $this->messenger->addStatus($this->t('The email address %email has been confirmed.', [
          '%email' => $email->getEmail(),
        ]));
      }
    }
    else {
      $this->emailConfirmer->confirm($email);
      $this->messenger->addError($this->t('You have exhausted your allowed attempts at confirming this email address. A new confirmation code has been sent.'));
    }

    // Redirect to the manage page.
    /** @noinspection PhpMethodParametersCountMismatchInspection */
    return (new CacheableRedirectResponse($url->toString()))
      ->addCacheableDependency($email)
      ->addCacheableDependency($user);
  }

}
