<?php

namespace Drupal\multiple_email;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Interface for multiple_email entities.
 */
interface EmailInterface extends ContentEntityInterface, EntityOwnerInterface {

  /**
   * The email address is unconfirmed.
   */
  const int UNCONFIRMED = 0;

  /**
   * The email address is confirmed.
   */
  const int CONFIRMED = 1;

  /**
   * The email address confirmation is canceled.
   */
  const int CANCELLED = 2;

  /**
   * Gets the email address.
   *
   * @return string
   *   The email address.
   */
  public function getEmail(): string;

  /**
   * Sets the email address.
   *
   * @param string $email
   *   The email address.
   *
   * @return $this
   */
  public function setEmail(string $email): static;

  /**
   * Gets the time that the email address was registered.
   *
   * @return int
   *   The timestamp of when the email address was registered.
   */
  public function getRegisteredTime(): int;

  /**
   * Gets the confirmation code.
   *
   * @return string
   *   The confirmation code.
   */
  public function getConfirmationCode(): string;

  /**
   * Sets the confirmation code.
   *
   * @param string $code
   *   The confirmation code.
   *
   * @return $this
   */
  public function setConfirmationCode(string $code): static;

  /**
   * Gets the time that the confirmation code was generated.
   *
   * @return int
   *   The timestamp of when the confirmation code was generated.
   */
  public function getCodeGeneratedTime(): int;

  /**
   * Sets the time that the confirmation code was generated.
   *
   * @param int $generated
   *   The timestamp of when the confirmation code was generated.
   *
   * @return $this
   */
  public function setCodeGeneratedTime(int $generated): static;

  /**
   * Returns the number of attempts for confirming the email address.
   *
   * @return int
   *   The number of attempts for confirming the email address.
   */
  public function getAttempts(): int;

  /**
   * Sets the number of attempts to confirm the email address.
   *
   * @param int $attempts
   *   The number of attempts to confirm the email address.
   *
   * @return $this
   */
  public function setAttempts(int $attempts): static;

  /**
   * Increments the number of attempts to confirm the email address.
   *
   * @return $this
   */
  public function incrementAttempts(): static;

  /**
   * Returns the status of the email address.
   *
   * @return int
   *   The status of the email address.
   */
  public function getStatus(): int;

  /**
   * Determines whether the email address confirmation has been canceled.
   *
   * @return bool
   *   TRUE if the email address confirmation is canceled.
   */
  public function isCancelled(): bool;

  /**
   * Determines whether the email address is confirmed.
   *
   * @return bool
   *   TRUE if the email address is confirmed.
   */
  public function isConfirmed(): bool;

  /**
   * Determines whether the email address is unconfirmed.
   *
   * @return bool
   *   TRUE if the email address is unconfirmed.
   */
  public function isUnconfirmed(): bool;

  /**
   * Determines whether the email address confirmation is pending.
   *
   * @return bool
   *   TRUE if the email address confirmation is pending.
   */
  public function isPending(): bool;

  /**
   * Sets the status of the email address.
   *
   * @param int $status
   *   The status of the email address.
   *
   * @return $this
   */
  public function setStatus(int $status): static;

  /**
   * Sets the email address status to confirmed.
   *
   * @return $this
   */
  public function setConfirmed(): static;

  /**
   * Sets the email address status to unconfirmed.
   *
   * @return $this
   */
  public function setUnconfirmed(): static;

  /**
   * Determines whether the email address is the primary one.
   */
  public function isPrimary(): bool;

  /**
   * Determines whether the account is the entity owner.
   *
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to verify.
   *
   * @return bool
   *   TRUE if the account is the owner, FALSE otherwise.
   */
  public function isOwner(AccountInterface $account): bool;

}
