<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\multiple_email\Handler\MultipleEmailAccessControlHandler;
use Drupal\multiple_email\EmailInterface;
use Drupal\multiple_email\EmailStorage;
use Drupal\multiple_email\EmailViewsData;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the multiple_email entity.
 */
#[ContentEntityType(
  id: 'multiple_email',
  label: new TranslatableMarkup('Email address'),
  label_collection: new TranslatableMarkup('Email addresses'),
  label_singular: new TranslatableMarkup('Email address'),
  label_plural: new TranslatableMarkup('Email addresses'),
  entity_keys: [
    'id' => 'eid',
    'label' => 'email',
    'langcode' => 'langcode',
    'uuid' => 'uuid',
    'status' => 'status',
    'uid' => 'uid',
    'owner' => 'uid',
  ],
  handlers: [
    'access' => MultipleEmailAccessControlHandler::class,
    'storage' => EmailStorage::class,
    'views_data' => EmailViewsData::class,
  ],
  admin_permission: 'administer multiple emails',
  base_table: 'multiple_email',
  label_count: [
    'singular' => '@count email address',
    'plural' => '@count email addresses',
  ],
)]
class Email extends ContentEntityBase implements EmailInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->getEmail();
  }

  /**
   * {@inheritdoc}
   */
  public function getEmail(): string {
    return $this->get('email')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setEmail(string $email): static {
    $this->set('email', $email);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegisteredTime(): int {
    return $this->get('time_registered')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmationCode(): string {
    return $this->get('confirm_code')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmationCode(string $code): static {
    $this->set('confirm_code', $code);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCodeGeneratedTime(): int {
    return $this->get('time_code_generated')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCodeGeneratedTime(int $generated): static {
    $this->set('time_code_generated', $generated);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAttempts(): int {
    return (int) $this->get('attempts')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setAttempts(int $attempts): static {
    $this->set('attempts', $attempts);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function incrementAttempts(): static {
    $attempts = $this->getAttempts() + 1;
    $this->set('attempts', $attempts);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): int {
    return $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function isCancelled(): bool {
    return $this->getStatus() == self::CANCELLED;
  }

  /**
   * {@inheritdoc}
   */
  public function isConfirmed(): bool {
    return $this->getStatus() === self::CONFIRMED;
  }

  /**
   * {@inheritdoc}
   */
  public function isUnconfirmed(): bool {
    return $this->getStatus() !== self::CONFIRMED;
  }

  /**
   * {@inheritdoc}
   */
  public function isPending(): bool {
    return !$this->isConfirmed() && !$this->isCancelled();
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(int $status): static {
    $this->set('status', $status);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmed(): static {
    $this->set('status', self::CONFIRMED);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setUnconfirmed(): static {
    $this->set('status', self::UNCONFIRMED);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrimary(): bool {
    $owner = $this->getOwner();

    return $owner && $owner->getEmail() === $this->getEmail();
  }

  /**
   * {@inheritdoc}
   */
  public function isOwner(AccountInterface $account): bool {
    $owner_id = $this->getOwnerId();

    return $owner_id && $owner_id === $account->id();
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\Exception\UnsupportedEntityTypeDefinitionException
   *   The entity type definition is not supported.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['email'] = BaseFieldDefinition::create('email')
      ->setLabel(new TranslatableMarkup('Email'))
      ->setRequired(TRUE)
      ->addConstraint('UniqueField');

    $fields['time_registered'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Registered'));

    $fields['confirm_code'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Confirmation code'));

    $fields['time_code_generated'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Code generated'));

    $fields['attempts'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Attempts'));

    $fields['status'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setDefaultValue(self::UNCONFIRMED);

    return $fields;
  }

}
