<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Form;

use Drupal\Component\Serialization\Json;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailStorageInterface;
use Drupal\user\UserInterface;

/**
 * Form for managing an accounts email addresses.
 */
class ManageForm extends FormBase {

  use AutowireTrait;

  /**
   * The email storage.
   *
   * @var \Drupal\multiple_email\EmailStorageInterface
   */
  protected readonly EmailStorageInterface $storage;

  /**
   * Constructs a new \Drupal\multiple_email\Form\ManageForm instance.
   *
   * @param \Drupal\multiple_email\EmailConfirmerInterface $emailConfirmer
   *   The email confirmer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   An error happened when getting the entity storage.
   */
  public function __construct(
    protected readonly EmailConfirmerInterface $emailConfirmer,
    EntityTypeManagerInterface $entity_type_manager,
    MessengerInterface $messenger,
  ) {
    /** @var \Drupal\multiple_email\EmailStorageInterface $storage */
    $storage = $entity_type_manager->getStorage('multiple_email');
    $this->storage = $storage;

    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'multiple_email_manage';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = NULL): array {
    $header = [
      'email' => $this->t('Email'),
      'status' => $this->t('Status'),
      'operations' => $this->t('Operations'),
    ];

    $form['emails'] = [
      '#type' => 'table',
      '#header' => $header,
      '#attributes' => [
        'id' => 'multiple-email-table-wrapper',
      ],
    ];

    if ($user) {
      $form_state->set('multiple_email_owner', $user);

      foreach ($this->storage->loadByUser($user) as $email) {
        $operations = [];

        if ($email->isConfirmed()) {
          if ($email->isPrimary()) {
            $status_text = $this->t('Primary');
          }
          else {
            $status_text = $this->t('Confirmed');
            $operations['set-primary'] = [
              'title' => $this->t('Set Primary'),
              'weight' => 10,
              'url' => Url::fromRoute(
                'multiple_email.set_primary',
                ['multiple_email' => $email->id()],
              ),
              'attributes' => [
                'class' => ['use-ajax'],
                'data-dialog-type' => 'modal',
                'data-dialog-options' => Json::encode([
                  'width' => 'auto',
                ]),
              ],
            ];
          }
        }
        elseif ($email->isPending()) {
          $status_text = $this->t('Confirmation pending');
          $operations['resend'] = [
            'title' => $this->t('Resend confirmation'),
            'weight' => 15,
            'url' => Url::fromRoute(
              'multiple_email.resend',
              ['multiple_email' => $email->id()],
            ),
            'attributes' => [
              'class' => ['use-ajax'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => Json::encode([
                'width' => 'auto',
              ]),
            ],
          ];
          $operations['cancel'] = [
            'title' => $this->t('Cancel confirmation'),
            'weight' => 20,
            'url' => Url::fromRoute(
              'multiple_email.cancel_confirmation',
              ['multiple_email' => $email->id()],
            ),
            'attributes' => [
              'class' => ['use-ajax'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => Json::encode([
                'width' => 'auto',
              ]),
            ],
          ];
        }
        elseif ($email->isCancelled()) {
          $status_text = $this->t('Confirmation cancelled');
          $operations['resend'] = [
            'title' => $this->t('Resend confirmation'),
            'weight' => 15,
            'url' => Url::fromRoute(
              'multiple_email.resend',
              ['multiple_email' => $email->id()],
            ),
            'attributes' => [
              'class' => ['use-ajax'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => Json::encode([
                'width' => 'auto',
              ]),
            ],
          ];
        }
        else {
          $status_text = $this->t('Unconfirmed');
          $operations['confirm'] = [
            'title' => $this->t('Confirm'),
            'weight' => 5,
            'url' => Url::fromRoute(
              'multiple_email.confirm',
              ['multiple_email' => $email->id()],
            ),
            'attributes' => [
              'class' => ['use-ajax'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => Json::encode([
                'width' => 'auto',
              ]),
            ],
          ];
        }

        // The primary email address cannot be removed.
        if (!$email->isPrimary()) {
          $operations['remove'] = [
            'title' => $this->t('Remove'),
            'weight' => 25,
            'url' => Url::fromRoute(
              'multiple_email.remove',
              ['multiple_email' => $email->id()],
            ),
            'attributes' => [
              'class' => ['use-ajax'],
              'data-dialog-type' => 'modal',
              'data-dialog-options' => Json::encode([
                'width' => 'auto',
              ]),
            ],
          ];
        }

        $form['emails'][$email->id()] = [
          'email' => [
            '#type' => 'markup',
            '#markup' => $email->getEmail(),
          ],
          'status' => [
            '#type' => 'markup',
            '#markup' => $status_text,
          ],
          'operations' => [
            '#type' => 'operations',
            '#links' => $operations,
          ],
        ];
      }
    }

    $form['emails']['add'] = [
      'email' => [
        '#type' => 'email',
        '#placeholder' => $this->t('Add a new email address'),
      ],
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Add email address'),
      ],
      'operations' => [],
    ];

    $action_url = Url::fromRoute('multiple_email.manage', [
      'user' => $user->id(),
    ]);

    $form['#action'] = $action_url->toString();
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue(['emails', 'add']);
    $email_address = $values['email'];

    if ($this->storage->loadByEmail($email_address)) {
      $message = $this->t('The email address @email has already been registered.', [
        '@email' => $email_address,
      ]);
      $form_state->setErrorByName('email', $message);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred while saving the multiple_email entity.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $emails = $form_state->getValue('emails');
    $email = reset($emails);
    $email_address = $email['email'];

    /** @var \Drupal\multiple_email\EmailInterface $email */
    $email = $this->storage->create([
      'email' => $email_address,
    ]);

    /** @var \Drupal\user\UserInterface $user */
    if ($user = $form_state->get('multiple_email_owner')) {
      $email->setOwner($user);
    }

    $email->save();
    $this->emailConfirmer->confirm($email);

    $message = $this->t('The email address @email has been added.', ['@email' => $email_address]);
    $this->messenger()->addMessage($message);

    $form_state->setRedirect('multiple_email.manage', [
      'user' => $email->getOwnerId(),
    ]);
  }

}
