<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUnused */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpUndefinedNamespaceInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Form;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form builder for the admin settings page.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->setModuleHandler($container->get('module_handler'));

    return $instance;
  }

  /**
   * Sets the module handler.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  protected function setModuleHandler(ModuleHandlerInterface $module_handler): void {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'multiple_email_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['multiple_email.settings', 'multiple_email.mail'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $config_settings = $this->config('multiple_email.settings');
    $config_mail = $this->config('multiple_email.mail');
    $token_help = $this->t('Available variables are: [site:name], [site:url], [site:login-url], [user:display-name], [user:account-name], [multiple_email:email], [multiple_email:confirm_code], [multiple_email:confirm_url], [multiple_email:confirm_deadline].');

    $token_enabled = $this->moduleHandler->moduleExists('token');
    if ($token_enabled) {
      $token_help = $this->t('The list of available tokens that can be used in emails is provided below.');
    }

    $form['hide_field'] = [
      '#type' => 'select',
      '#title' => t('Hide the email field'),
      '#description' => t('Hides the email field when editing a user account'),
      '#options' => ['No', 'Yes'],
      '#default_value' => $config_settings->get('hide_field'),
    ];

    $form['edit_emails'] = [
      '#type' => 'select',
      '#title' => t('Allow email editing'),
      '#description' => t('Allows editing email addresses. It is equivalent to deleting and adding a new email address, as edited emails must be re-confirmed. If enabled, email addresses (excluding the primary email) may be edited via the multiple email tab.'),
      '#options' => ['No', 'Yes'],
      '#default_value' => $config_settings->get('edit_emails'),
    ];

    $form['confirm_attempts'] = [
      '#type' => 'textfield',
      '#size' => 4,
      '#title' => t('Confirmation attempts'),
      '#description' => t('The number of times a confirmation code can be used before a new one is generated. If set to 0, no new codes are sent after the first one.'),
      '#default_value' => $config_settings->get('confirm.attempts'),
    ];

    $form['confirm_deadline'] = [
      '#type' => 'textfield',
      '#size' => 4,
      '#title' => t('Confirmation days'),
      '#description' => t('The number of days before a confirmation code expires. If 0, confirmations do not expire.'),
      '#default_value' => $config_settings->get('confirm.deadline'),
    ];

    $form['email'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Emails'),
    ];

    $form['email_confirmation'] = [
      '#type' => 'details',
      '#title' => $this->t('Confirmation'),
      '#description' => $this->t('Customize the message sent when a new email is added.') . ' ' . $token_help,
      '#group' => 'email',
    ];

    $form['email_confirmation']['confirmation_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config_mail->get('confirmation.subject'),
    ];

    $form['email_confirmation']['confirmation_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config_mail->get('confirmation.body'),
    ];

    $form['email_expiration'] = [
      '#type' => 'details',
      '#title' => $this->t('Expiration'),
      '#description' => $this->t('Customize the message sent when an unconfirmed email address expires.') . ' ' . $token_help,
      '#group' => 'email',
    ];

    $form['email_expiration']['expiration_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config_mail->get('expiration.subject'),
    ];

    $form['email_expiration']['expiration_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config_mail->get('expiration.body'),
    ];

    if ($token_enabled) {
      $form['email']['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['user', 'multiple_email'],
        '#show_restricted' => TRUE,
        '#show_nested' => FALSE,
        '#weight' => 90,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (!is_numeric($form_state->getValue('confirm_attempts'))) {
      $form_state->setErrorByName('confirm_attempts', $this->t('Confirmation attempts must be an number!'));
    }

    if (!is_numeric($form_state->getValue('confirm_deadline'))) {
      $form_state->setErrorByName('confirm_deadline', $this->t('Confirmation days must be an number!'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('multiple_email.settings')
      ->set('hide_field', (bool) $form_state->getValue('hide_field'))
      ->set('edit_emails', (bool) $form_state->getValue('edit_emails'))
      ->set('confirm.attempts', (int) $form_state->getValue('confirm_attempts'))
      ->set('confirm.deadline', (int) $form_state->getValue('confirm_deadline'))
      ->save();

    $this->config('multiple_email.mail')
      ->set('confirmation.subject', $form_state->getValue('confirmation_subject'))
      ->set('confirmation.body', $form_state->getValue('confirmation_body'))
      ->set('expiration.subject', $form_state->getValue('expiration_subject'))
      ->set('expiration.body', $form_state->getValue('expiration_body'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
