<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUnused, PhpUnusedParameterInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Hook;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\PluralTranslatableMarkup;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslationManager;
use Drupal\Core\Url;
use Drupal\multiple_email\EmailInterface;
use Drupal\user\UserInterface;

/**
 * Token hook implementations.
 */
class MultipleEmailTokenHooks {

  use StringTranslationTrait;

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Hook\MultipleEmailTokenHooks instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\StringTranslation\TranslationManager $translationManager
   *   The translation manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly TranslationManager $translationManager,
    TranslationInterface $translation,
  ) {
    $this->setStringTranslation($translation);
  }

  // phpcs:enable

  /**
   * Implements hook_token_info().
   */
  public function tokenInfo(): array {
    $type = [
      'name' => $this->t('Multiple email'),
      'description' => $this->t('Tokens related to multiple email addresses.'),
      'needs-data' => 'multiple_email',
    ];

    $multiple_email = [
      'email' => [
        'name' => $this->t('Email'),
        'description' => $this->t('The email address.'),
      ],
      'confirm_code' => [
        'name' => $this->t('Confirmation code'),
        'description' => $this->t('The code to confirm the email address.'),
      ],
      'confirm_deadline' => [
        'name' => $this->t('Confirmation deadline'),
        'description' => $this->t('The deadline for confirming the email address.'),
      ],
      'confirm_url' => [
        'name' => $this->t('Confirmation URL'),
        'description' => $this->t('The URL to confirm the email address.'),
      ],
    ];

    return [
      'types' => [
        'multiple_email' => $type,
      ],
      'tokens' => [
        'multiple_email' => $multiple_email,
      ],
    ];
  }

  /**
   * Implements hook_tokens().
   */
  public function tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
    $replacements = [];

    if ($type == 'multiple_email' && !empty($data['multiple_email'])) {
      /** @var \Drupal\multiple_email\EmailInterface $email */
      $email = $data['multiple_email'];

      foreach ($tokens as $name => $original) {
        $replacements[$original] = match ($name) {
          'email' => $email->getEmail(),
          'confirm_code' => $email->getConfirmationCode(),
          'confirm_deadline' => $this->getConfirmDeadline(),
          'confirm_url' => $this->getConfirmUrl($email),
          default => '',
        };
      }
    }

    return $replacements;
  }

  /**
   * Creates a unique hash value for the confirmation code.
   *
   * @param string $code
   *   The confirmation code.
   * @param \Drupal\user\UserInterface $account
   *   The user account.
   *
   * @return string
   *   The hash value.
   */
  protected function hashCode(string $code, UserInterface $account): string {
    return Crypt::hmacBase64($code, Settings::getHashSalt() . $account->getPassword());
  }

  /**
   * Returns the confirmation deadline.
   *
   * @return \Drupal\Core\StringTranslation\PluralTranslatableMarkup|\Drupal\Core\StringTranslation\TranslatableMarkup
   *   The confirmation deadline.
   */
  protected function getConfirmDeadline(): PluralTranslatableMarkup|TranslatableMarkup {
    $confirm = $this->configFactory
      ->get('multiple_email.settings')
      ->get('confirm');

    if ($confirm['deadline']) {
      $days = $this->translationManager->formatPlural($confirm['deadline'], '1 day', '@count days');
    }
    else {
      $days = $this->t('unlimited days');
    }

    return $days;
  }

  /**
   * Gets the confirmation URL.
   *
   * @param \Drupal\multiple_email\EmailInterface $email
   *   The email.
   *
   * @return string
   *   The confirmation URL.
   */
  protected function getConfirmUrl(EmailInterface $email): string {
    $account = $email->getOwner();
    $destination = Url::fromRoute('multiple_email.confirm_user', [
      'user' => $email->getOwnerId(),
      'multiple_email' => $email->id(),
      'code' => $this->hashCode($email->getConfirmationCode(), $account),
    ]);
    $url = Url::fromRoute('user.login')
      ->setOption('query', ['destination' => $destination->toString()])
      ->setAbsolute();

    return $url->toString();
  }

}
