<?php

// phpcs:disable Drupal.WhiteSpace.OpenTagNewline.BlankLine
// phpcs:disable SlevomatCodingStandard.TypeHints.DeclareStrictTypes.IncorrectWhitespaceBeforeDeclare
// phpcs:disable Drupal.Commenting.DocComment.MissingShort
// phpcs:disable Drupal.Commenting.FileComment.NamespaceNoFileDoc
// phpcs:disable Drupal.Commenting.DocComment.ContentAfterOpen
/** @noinspection PhpUnused */
/** @noinspection PhpUnusedParameterInspection */
/** @noinspection PhpUndefinedNamespaceInspection */
/** @noinspection PhpUndefinedClassInspection */
/** @noinspection PhpFullyQualifiedNameUsageInspection */

declare(strict_types=1);

// phpcs:enable

namespace Drupal\Tests\multiple_email\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\multiple_email\Entity\Email;
use Drupal\multiple_email\Handler\MultipleEmailAccessControlHandler;
use Drupal\Tests\user\Traits\UserCreationTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the multiple_email entity access handler.
 */
#[CoversClass(MultipleEmailAccessControlHandler::class)]
#[Group('multiple_email')]
#[RunTestsInSeparateProcesses]
class MultipleEmailAccessControlHandlerTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'multiple_email',
    'system',
    'user',
  ];

  /**
   * The accounts used for the test.
   *
   * @var \Drupal\user\UserInterface[]
   */
  protected array $accounts;

  /**
   * The multiple_email entity used for testing.
   *
   * @var \Drupal\multiple_email\Entity\Email
   */
  protected Email $emailEntity;

  /**
   * The access control handler to test.
   */
  protected MultipleEmailAccessControlHandler $accessHandler;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   The entity type plugin for the multiple_email entity was not found.
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurrent when creating the test accounts.
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('multiple_email');
    $this->installConfig('multiple_email');

    // Create first the account whose ID is 1, so other accounts do not have all
    // the permissions.
    /** @noinspection PhpRedundantOptionalArgumentInspection */
    $this->createUser([], values: ['uid' => 1]);

    $this->accounts[0] = $this->createUser(['administer multiple emails']);
    $this->accounts[1] = $this->createUser(['use multiple emails']);
    $this->accounts[2] = $this->createUser(['use multiple emails']);
    $this->accounts[3] = $this->createUser(['access content']);
    $this->emailEntity = Email::create(
      ['email' => $this->randomMachineName() . '@example.com'],
    );
    $this->accessHandler = new MultipleEmailAccessControlHandler(
      \Drupal::entityTypeManager()->getDefinition('multiple_email'),
      \Drupal::translation()
    );

    $this->emailEntity->save();
  }

  /**
   * Tests the operations available for an unconfirmed multiple_email entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred when saving an entity.
   *
   * @noinspection PhpUndefinedMethodInspection
   */
  public function testUnconfirmedEmail() {
    $this->emailEntity->setOwner($this->accounts[1])
      ->setUnconfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The multiple_email entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isUnconfirmed(), 'The multiple_email entity is unconfirmed.');

    // An account with the "use multiple emails" permission can access the
    // multiple_email entities associated to it.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[1]), 'The account #1 can view the multiple_email entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[1]), 'The account #1 can edit the multiple_email entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[1]), 'The account #1 can delete the multiple_email entity.');

    // An account without the "use multiple emails" nor the
    // "administer multiple emails" permission cannot access multiple_email
    // entities associated to other accounts.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[3]), 'The account #3 cannot view the multiple_email entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[3]), 'The account #3 cannot edit the multiple_email entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[3]), 'The account #3 cannot delete the multiple_email entity.');

    // An account with the "use multiple emails" permission cannot access
    // multiple_email entities associated to other accounts.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[2]), 'The account #2 cannot view the multiple_email entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[2]), 'The account #2 cannot edit the multiple_email entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[2]), 'The account #2 cannot delete the multiple_email entity.');

    // An account with the "administer multiple emails" permission can access
    // multiple_email entities associated to other accounts.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[0]), 'The account #0 can view the multiple_email entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[0]), 'The account #0 can edit the multiple_email entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[0]), 'The account #0 can delete the multiple_email entity.');

    // A new confirmation email can only be requested by the owner and the admin
    // accounts.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[0]), 'The account #0 can request a new confirmation email.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[1]), 'The account #1 can request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[2]), 'The account #2 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[3]), 'The account #3 cannot request a new confirmation email.');

    // Unconfirmed multiple_email entities cannot be set as primary email.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[0]), 'The account #0 cannot set the multiple_email entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[1]), 'The account #1 cannot set the multiple_email entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[2]), 'The account #2 cannot set the multiple_email entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[3]), 'The account #3 cannot set the multiple_email entity as primary.');
  }

  /**
   * Tests the operations available for a confirmed multiple_email entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred when saving an entity.
   *
   * @noinspection PhpUndefinedMethodInspection
   */
  public function testConfirmedEmail() {
    $this->emailEntity->setOwner($this->accounts[1])
      ->setConfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The multiple_email entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isConfirmed(), 'The multiple_email entity is confirmed.');

    // A new confirmation email cannot be requested for confirmed multiple_email
    // entities.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[0]), 'The account #0 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[1]), 'The account #1 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[2]), 'The account #2 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[3]), 'The account #3 cannot request a new confirmation email.');

    // Confirmed multiple_email entities can be set as primary email.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[0]), 'The account #0 can set the multiple_email entity as primary.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[1]), 'The account #1 can set the multiple_email entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[2]), 'The account #2 cannot set the multiple_email entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[3]), 'The account #3 cannot set the multiple_email entity as primary.');
  }

  /**
   * Tests that the primary email can only be viewed.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error happened when saving an entity.
   *
   * @noinspection PhpUndefinedMethodInspection
   */
  public function testPrimaryEmail() {
    $operations = [
      'cancel',
      'confirm',
      'delete',
      'edit',
      'remove',
      'resend_confirmation',
      'set_primary',
      'update',
    ];
    $tested_operation = 'view';

    $this->accounts[1]->setEmail($this->emailEntity->getEmail())->save();
    $this->emailEntity->setOwner($this->accounts[1])
      ->setConfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The multiple_email entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isConfirmed(), 'The multiple_email entity is confirmed.');
    $this->assertTrue($this->emailEntity->isPrimary(), 'The multiple_email entity is a primary email.');

    // Test that admin and owner account can view the multiple_email entity.
    $this->assertTrue((bool) $this->accessHandler->access($this->emailEntity, $tested_operation, $this->accounts[0]), 'Account #0 can view the entity');
    $this->assertTrue((bool) $this->accessHandler->access($this->emailEntity, $tested_operation, $this->accounts[1]), 'Account #1 can view the entity');

    // Test that other operations are not allowed for any account.
    foreach ($operations as $operation) {
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[0]), sprintf('The %s operation is not allowed for the account #0', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[1]), sprintf('The %s operation is not allowed for the account #1', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[2]), sprintf('The %s operation is not allowed for the account #2', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[3]), sprintf('The %s operation is not allowed for the account #3', $operation));
    }
  }

  /**
   * Tests multiple_email entities can be created by the right accounts.
   *
   * @noinspection PhpUndefinedMethodInspection
   */
  public function testCreateAccess() {
    // Only accounts with either the "use multiple emails" or the
    // "administer multiple emails" permission can create multiple_email
    // entities.
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[0]), 'The account #0 can create a multiple_email entity.');
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[1]), 'The account #1 can create a multiple_email entity.');
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[2]), 'The account #2 can create a multiple_email entity.');
    $this->assertFalse($this->accessHandler->createAccess('multiple_email', $this->accounts[3]), 'The account #3 cannot create a multiple_email entity.');
  }

}
