<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Cache\CacheableRedirectResponse;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailInterface;
use Drupal\user\UserInterface;

/**
 * Controller for email address entities.
 */
class EmailController implements ContainerInjectionInterface {

  use AutowireTrait;
  use StringTranslationTrait;

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Controller\EmailController instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\multiple_email\EmailConfirmerInterface $emailConfirmer
   *   The email confirmer.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EmailConfirmerInterface $emailConfirmer,
    protected readonly TimeInterface $time,
    protected readonly MessengerInterface $messenger,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  // phpcs:enable

  /**
   * Validates the path parameters.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user account.
   * @param \Drupal\multiple_email\EmailInterface $multiple_email
   *   The email.
   * @param string $code
   *   The hashed code.
   *
   * @return bool
   *   TRUE if the path parameters are valid, FALSE otherwise.
   */
  protected function validatePathParameters(UserInterface $user, EmailInterface $multiple_email, string $code): bool {
    return hash_equals(Crypt::hmacBase64($code, Settings::getHashSalt() . $user->getPassword()), $code);
  }

  /**
   * Gets the title for the cancel confirmation form.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title.
   */
  public function cancelConfirmationTitle(string $entity_type_id): TranslatableMarkup {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    return $this->t('Cancel @entity-type confirmation', [
      '@entity-type' => $entity_type->getSingularLabel(),
    ]);
  }

  /**
   * Gets the title for the confirm form.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title.
   */
  public function confirmTitle(string $entity_type_id): TranslatableMarkup {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    return $this->t('Confirm @entity-type', [
      '@entity-type' => $entity_type->getSingularLabel(),
    ]);
  }

  /**
   * Gets the title for the remove form.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title.
   */
  public function removeTitle(string $entity_type_id): TranslatableMarkup {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    return $this->t('Remove @entity-type', [
      '@entity-type' => $entity_type->getSingularLabel(),
    ]);
  }

  /**
   * Gets the title for the resend form.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title.
   */
  public function resendTitle(string $entity_type_id): TranslatableMarkup {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    return $this->t('Resend @entity-type confirmation', [
      '@entity-type' => $entity_type->getSingularLabel(),
    ]);
  }

  /**
   * Gets the title for the set primary form.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title.
   */
  public function setPrimaryTitle(string $entity_type_id): TranslatableMarkup {
    $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);

    return $this->t('Set @entity-type as primary', [
      '@entity-type' => $entity_type->getSingularLabel(),
    ]);
  }

  /**
   * Confirms the email address.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   The email address entity cannot be saved.
   */
  public function confirmUser(UserInterface $user, EmailInterface $email, string $code): CacheableRedirectResponse {
    $attempts = $email->getAttempts();
    $confirmation_settings = $this->configFactory
      ->get('multiple_email.settings')
      ->get('confirm');
    $url = Url::fromRoute(
      'multiple_email.manage',
      ['user' => $user->id()],
      ['absolute' => TRUE],
    );

    if ($attempts < $confirmation_settings['attempts']) {
      if ($this->validatePathParameters($user, $email, $code)) {
        $email->setAttempts($attempts + 1)
          ->save();

        $this->messenger->addError($this->t('The confirmation code was incorrect.'));
      }
      else {
        $email->setConfirmed()
          ->setConfirmationTime($this->time->getRequestTime())
          ->save();
        $this->messenger->addStatus($this->t('The email address %email has been confirmed.', [
          '%email' => $email->getEmail(),
        ]));
      }
    }
    else {
      $this->emailConfirmer->confirm($email);
      $this->messenger->addError($this->t('You have exhausted your allowed attempts at confirming this email address. A new confirmation code has been sent.'));
    }

    // Redirect to the manage page.
    return (new CacheableRedirectResponse($url->toString()))
      ->addCacheableDependency($email)
      ->addCacheableDependency($user);
  }

}
