<?php

declare(strict_types=1);

namespace Drupal\multiple_email;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Utility\Token;
use Random\Randomizer;

/**
 * Confirms an email address.
 */
readonly class EmailConfirmer implements EmailConfirmerInterface {

  /**
   * Constructs a new \Drupal\multiple_email\EmailConfirmer instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The datetime service.
   * @param \Drupal\Core\Mail\MailManagerInterface $mailManager
   *   The mail manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected TimeInterface $time,
    protected MailManagerInterface $mailManager,
    protected LanguageManagerInterface $languageManager,
    protected Token $token,
    protected LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   The email address entity cannot be saved.
   */
  public function confirm(EmailInterface $email): void {
    $email
      ->setConfirmationCode($this->generateCode())
      ->setCodeGeneratedTime($this->time->getRequestTime())
      ->setAttempts(0)
      ->setUnconfirmed()
      ->save();

    $this->send($email, 'confirmation');
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   The email address entity cannot be saved.
   */
  public function cancelConfirmation(EmailInterface $email): void {
    $email->setStatus(EmailInterface::CANCELLED)
      ->save();
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   The email address entity cannot be deleted.
   */
  public function expire(EmailInterface $email): void {
    $email->delete();

    $this->send($email, 'expiration');
  }

  /**
   * Generates a random string of given length from given characters.
   *
   * If no characters are specified, then it uses a-zA-Z0-9. Characters are
   * specified as a string containing every valid character. Duplicates will
   * (in theory) increase that character's chances of occurring in the random
   * string.
   *
   * @param int $length
   *   Length of the random code. Defaults to 10 characters.
   * @param string $chars
   *   The characters to use in the code.
   *
   * @return string
   *   The generated code.
   */
  protected function generateCode(int $length = 10, string $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUBWXYZ0123456789'): string {
    return (new Randomizer())->getBytesFromString($chars, $length);
  }

  /**
   * Sends a message to the user.
   *
   * @param \Drupal\multiple_email\EmailInterface $email
   *   The email address to send the message to.
   * @param string $type
   *   The type of email to send, either confirmation or expiration.
   */
  protected function send(EmailInterface $email, string $type): void {
    $settings = $this->configFactory->get('multiple_email.mail')
      ->get($type);
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $variables = [
      'user' => $email->getOwner(),
      'multiple_email' => $email,
    ];
    $token_options = [
      'langcode' => $langcode,
      'clear' => TRUE,
    ];
    $to = $email->getEmail();
    $subject = $this->token->replace($settings['subject'], $variables, $token_options);
    $body = $this->token->replace($settings['body'], $variables, $token_options);
    $params = [
      'subject' => $subject,
      'body' => $body,
    ];
    $result = $this->mailManager->mail('multiple_email', 'confirmation', $to, $langcode, $params);
    $args = [
      '@type' => $type,
      '@email' => $to,
    ];

    if ($result['result']) {
      $this->logger->notice('The @type message has been sent to @email.', $args);
    }
    else {
      $this->logger->error('The @type message has not been sent to @email.', $args);
    }
  }

}
