<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\multiple_email\EmailInterface;
use Drupal\multiple_email\EmailStorage;
use Drupal\multiple_email\EmailViewsData;
use Drupal\multiple_email\Form\ConfirmCancelForm;
use Drupal\multiple_email\Form\ConfirmConfirmForm;
use Drupal\multiple_email\Form\ConfirmDeleteForm;
use Drupal\multiple_email\Form\ConfirmResendForm;
use Drupal\multiple_email\Form\ConfirmSetPrimaryForm;
use Drupal\multiple_email\Handler\MultipleEmailAccessControlHandler;
use Drupal\multiple_email\Handler\MultipleEmailListBuilder;
use Drupal\user\EntityOwnerTrait;
use Drupal\user\UserInterface;

/**
 * Defines the email address entity.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
#[ContentEntityType(
  id: 'multiple_email',
  label: new TranslatableMarkup('Email address'),
  label_collection: new TranslatableMarkup('Email addresses'),
  label_singular: new TranslatableMarkup('Email address'),
  label_plural: new TranslatableMarkup('Email addresses'),
  entity_keys: [
    'id' => 'eid',
    'label' => 'email',
    'langcode' => 'langcode',
    'uuid' => 'uuid',
    'status' => 'status',
    'uid' => 'uid',
    'owner' => 'uid',
  ],
  handlers: [
    'access' => MultipleEmailAccessControlHandler::class,
    'form' => [
      'cancel_confirmation' => ConfirmCancelForm::class,
      'confirm' => ConfirmConfirmForm::class,
      'remove' => ConfirmDeleteForm::class,
      'resend' => ConfirmResendForm::class,
      'set_primary' => ConfirmSetPrimaryForm::class,
    ],
    'list_builder' => MultipleEmailListBuilder::class,
    'route_provider' => [
      'html' => MultipleEmailRouteProvider::class,
    ],
    'storage' => EmailStorage::class,
    'views_data' => EmailViewsData::class,
  ],
  links: [
    'cancel_confirmation' => '/multiple-email/cancel-confirmation/{multiple_email}',
    'collection' => '/admin/people/multiple_email',
    'confirm' => '/multiple-email/confirm/{multiple_email}',
    'confirm_user' => '/multiple-email/user/{user}/confirm/{multiple_email}/code/{code}',
    'remove' => '/multiple-email/remove/{multiple_email}',
    'resend' => '/multiple-email/resend/{multiple_email}',
    'set_primary' => '/multiple-email/set-primary/{multiple_email}',
  ],
  admin_permission: 'administer multiple emails',
  base_table: 'multiple_email',
  label_count: [
    'singular' => '@count email address',
    'plural' => '@count email addresses',
  ],
)]
class Email extends ContentEntityBase implements EmailInterface {

  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function get($field_name): FieldItemListInterface {
    $deprecated_fields = [
      'time_registered' => [
        'The time_registered field is deprecated in multiple_email:3.0.0 and is removed from multiple_email:4.0.0. Use the created field instead. See https://www.drupal.org/node/3565541',
        'created',
      ],
    ];

    if (isset($deprecated_fields[$field_name])) {
      @trigger_error($deprecated_fields[$field_name][0], E_USER_DEPRECATED);

      $field_name = $deprecated_fields[$field_name][1];
    }

    return parent::get($field_name);
  }

  /**
   * {@inheritdoc}
   */
  public function set($name, $value, $notify = TRUE): static {
    $deprecated_fields = [
      'time_registered' => [
        'The time_registered field is deprecated in multiple_email:3.0.0 and is removed from multiple_email:4.0.0. Use the created field instead. See https://www.drupal.org/node/3565541',
        'created',
        NULL,
      ],
    ];

    if (isset($deprecated_fields[$name])) {
      @trigger_error($deprecated_fields[$name][0], E_USER_DEPRECATED);

      if (!empty($deprecated_fields[$name][2])) {
        $callable = $deprecated_fields[$name][2];
        $value = $callable($name, $deprecated_fields[$name][1], $value);
        $name = $deprecated_fields[$name][1];
      }
    }

    parent::set($name, $value);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function hasField($field_name): bool {
    $deprecated_fields = [
      'time_registered',
    ];

    return isset($deprecated_fields[$field_name]) || parent::hasField($field_name);
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->getEmail();
  }

  /**
   * {@inheritdoc}
   */
  public function getEmail(): string {
    return $this->get('email')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setEmail(string $email): static {
    $this->set('email', $email);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRegisteredTime(): int {
    @trigger_error('The ' . __METHOD__ . ' method is deprecated in multiple_email:3.0.0 and is removed from multiple_email:4.0.0. Use ::getCreatedTime() instead. See https://www.drupal.org/node/3565541', E_USER_DEPRECATED);

    return $this->getCreatedTime();
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    $created = (int) $this->get('created')->value;

    if (empty($created)) {
      // The field has not been initialized, yet. Set it to the time the owner
      // account was created.
      $created = $this->getOwner()->getCreatedTime();

      $this->set('created', $created);
    }

    return $created;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmationCode(): string {
    return $this->get('confirm_code')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmationCode(string $code): static {
    $this->set('confirm_code', $code);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmationTime(): int {
    return (int) $this->get('confirmation_time')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmationTime(int $time): static {
    $this->set('confirmation_time', $time);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCodeGeneratedTime(): int {
    return (int) $this->get('time_code_generated')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCodeGeneratedTime(int $generated): static {
    $this->set('time_code_generated', $generated);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAttempts(): int {
    return (int) $this->get('attempts')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setAttempts(int $attempts): static {
    $this->set('attempts', $attempts);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function incrementAttempts(): static {
    $attempts = $this->getAttempts() + 1;
    $this->set('attempts', $attempts);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): int {
    return (int) $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusLabel(): TranslatableMarkup {
    if ($this->isConfirmed()) {
      $label = $this->isPrimary() ? new TranslatableMarkup('Primary') : new TranslatableMarkup('Confirmed');
    }
    elseif ($this->isPending()) {
      $label = new TranslatableMarkup('Confirmation pending');
    }
    elseif ($this->isCancelled()) {
      $label = new TranslatableMarkup('Confirmation cancelled');
    }
    else {
      $label = new TranslatableMarkup('Unconfirmed');
    }

    return $label;
  }

  /**
   * {@inheritdoc}
   */
  public function isCancelled(): bool {
    return $this->getStatus() == self::CANCELLED;
  }

  /**
   * {@inheritdoc}
   */
  public function isConfirmed(): bool {
    return $this->getStatus() === self::CONFIRMED;
  }

  /**
   * {@inheritdoc}
   */
  public function isUnconfirmed(): bool {
    return $this->getStatus() === self::UNCONFIRMED;
  }

  /**
   * {@inheritdoc}
   */
  public function isPending(): bool {
    return !$this->isConfirmed() && !$this->isCancelled();
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus(int $status): static {
    $this->set('status', $status);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfirmed(): static {
    $this->set('status', self::CONFIRMED);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setUnconfirmed(): static {
    $this->set('status', self::UNCONFIRMED);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPrimary(): bool {
    $owner = $this->getOwner();

    return $owner && $owner->getEmail() === $this->getEmail();
  }

  /**
   * {@inheritdoc}
   */
  public function getPreviousOwnerId(): int {
    return $this->getEntityKey('previous_owner');
  }

  /**
   * {@inheritdoc}
   */
  public function setPreviousOwnerId(int $uid): static {
    $this->set('previous_owner', $uid);

    return $this;

  }

  /**
   * {@inheritdoc}
   */
  public function getPreviousOwner(): ?UserInterface {
    return $this->get('previous_owner')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setPreviousOwner(UserInterface $user): static {
    $this->set('previous_owner', $user);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerToAnonymous(): static {
    // Do not change the owner if it is already the anonymous account.
    if (!$this->isOwnerAnonymous()) {
      // Since the owner is changed to anonymous, the status must be set to
      // unconfirmed.
      $this->setPreviousOwnerId((int) $this->getOwnerId())
        ->setOwnerId(0)
        ->setStatus(self::UNCONFIRMED);
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isOwner(AccountInterface $account): bool {
    $owner = $this->getOwner();

    return $owner && $owner->id() === $account->id();
  }

  /**
   * {@inheritdoc}
   */
  public function isOwnerAnonymous(): bool {
    $owner = $this->getOwner();

    return $owner && $owner->isAnonymous();
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\Exception\UnsupportedEntityTypeDefinitionException
   *   The entity type definition is not supported.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['email'] = BaseFieldDefinition::create('email')
      ->setLabel(new TranslatableMarkup('Email'))
      ->setDescription(new TranslatableMarkup('The email address.'))
      ->setRequired(TRUE)
      ->addConstraint('UniqueField');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the email address was created.'));

    $fields['previous_owner'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Previous owner'))
      ->setDescription(new TranslatableMarkup('The user that had this email address before the current owner.'))
      ->setSetting('target_type', 'user')
      ->setTranslatable($entity_type->isTranslatable())
      ->setDefaultValue(0);

    $fields['confirm_code'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Confirmation code'))
      ->setDescription(t('The confirmation code.'));

    $fields['confirmation_time'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Confirmation time'))
      ->setDescription(new TranslatableMarkup('The time that the email was confirmed.'));

    $fields['time_code_generated'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(new TranslatableMarkup('Code generation time'))
      ->setDescription(new TranslatableMarkup('The time that the confirmation code was generated.'));

    $fields['attempts'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Attempts'))
      ->setDescription(new TranslatableMarkup('The number of attempts done to confirm the email address.'));

    $fields['status'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setDescription(new TranslatableMarkup('The status of the email address.'))
      ->setDefaultValue(self::UNCONFIRMED);

    return $fields;
  }

}
