<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Entity;

use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider;
use Drupal\multiple_email\Controller\EmailController;
use Symfony\Component\Routing\Route;

/**
 * Route provider for email address entities.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
class MultipleEmailRouteProvider extends DefaultHtmlRouteProvider {

  /**
   * {@inheritdoc}
   */
  public function getRoutes(EntityTypeInterface $entity_type) {
    /** @var \Symfony\Component\Routing\RouteCollection|\Symfony\Component\Routing\Route[] $collection */
    $collection = parent::getRoutes($entity_type);
    $entity_type_id = $entity_type->id();

    if ($cancel_confirmation_form_route = $this->getCancelConfirmFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.cancel_confirmation", $cancel_confirmation_form_route);
    }

    if ($confirm_form_route = $this->getConfirmFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.confirm", $confirm_form_route);
    }

    if ($confirm_user_route = $this->getConfirmUserRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.confirm_user", $confirm_user_route);
    }

    if ($remove_form_route = $this->getRemoveFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.remove", $remove_form_route);
    }

    if ($resend_form_route = $this->getResendFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.resend", $resend_form_route);
    }

    if ($set_primary_form_route = $this->getSetPrimaryFormRoute($entity_type)) {
      $collection->add("entity.{$entity_type_id}.set_primary", $set_primary_form_route);
    }

    return $collection;
  }

  /**
   * Gets the route to the cancel confirmation form.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getCancelConfirmFormRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'cancel_confirmation';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_entity_form' => "{$entity_type_id}.{$operation}",
        '_title_callback' => EmailController::class . '::cancelConfirmationTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.{$operation}",
        $entity_type_id => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
      ]);

      return $route;
    }

    return NULL;
  }

  /**
   * Gets the route to the confirmation form.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getConfirmFormRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'confirm';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_entity_form' => "{$entity_type_id}.{$operation}",
        '_title_callback' => EmailController::class . '::confirmTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.{$operation}",
        $entity_type_id => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
      ]);

      return $route;
    }

    return NULL;
  }

  /**
   * Gets the route to the page to confirm the email address added for a user.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getConfirmUserRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'confirm_user';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_controller' => EmailController::class . '::confirmUser',
        '_title_callback' => EmailController::class . '::confirmUserTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.confirm",
        $entity_type_id => '\\d+',
        'user' => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
        'user' => [
          'type' => 'entity:user',
        ],
      ]);
      $route->setOption('_admin_route', TRUE);

      return $route;
    }

    return NULL;
  }

  /**
   * Gets the route to the resend confirmation form.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getResendFormRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'resend';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_entity_form' => "{$entity_type_id}.{$operation}",
        '_title_callback' => EmailController::class . '::resendTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.{$operation}",
        $entity_type_id => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
      ]);

      return $route;
    }

    return NULL;
  }

  /**
   * Gets the route to the remove form.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getRemoveFormRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'remove';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_entity_form' => "{$entity_type_id}.{$operation}",
        '_title_callback' => EmailController::class . '::removeTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.{$operation}",
        $entity_type_id => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
      ]);

      return $route;
    }

    return NULL;
  }

  /**
   * Gets the route to the set primary form.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The route, if available.
   */
  protected function getSetPrimaryFormRoute(EntityTypeInterface $entity_type): ?Route {
    $operation = 'set_primary';

    if ($entity_type->hasLinkTemplate($operation)) {
      $entity_type_id = $entity_type->id();
      $route = new Route($entity_type->getLinkTemplate($operation));

      $route->setDefaults([
        '_entity_form' => "{$entity_type_id}.{$operation}",
        '_title_callback' => EmailController::class . '::setPrimaryTitle',
        'entity_type_id' => $entity_type_id,
      ]);
      $route->setRequirements([
        '_entity_access' => "{$entity_type_id}.{$operation}",
        $entity_type_id => '\\d+',
      ]);
      $route->setOption('parameters', [
        $entity_type_id => [
          'type' => "entity:{$entity_type_id}",
        ],
      ]);

      return $route;
    }

    return NULL;
  }

}
