<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailInterface;

/**
 * Base class for AJAX confirm forms.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
abstract class AjaxConfirmFormBase extends FormBase {

  use AutowireTrait;

  /**
   * The email address.
   *
   * @var \Drupal\multiple_email\EmailInterface
   */
  protected EmailInterface $email;

  /**
   * Constructs a new \Drupal\multiple_email\Form\AjaxConfirmFormBase instance.
   *
   * @param Drupal\multiple_email\EmailConfirmerInterface $emailConfirmer
   *   The email confirmer.
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly EmailConfirmerInterface $emailConfirmer,
    protected readonly FormBuilderInterface $formBuilder,
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?EmailInterface $multiple_email = NULL): array {
    $this->email = $multiple_email;

    $form['description'] = [
      '#markup' => $this->getDescription(),
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'button',
        '#button_type' => 'primary',
        '#value' => $this->t('Confirm'),
        '#ajax' => [
          'callback' => [$this, 'ajaxSubmit'],
        ],
      ],
      'cancel' => [
        '#type' => 'button',
        '#value' => $this->t('Cancel'),
        '#ajax' => [
          'callback' => [$this, 'closeDialog'],
        ],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->doAction();

    $form_state->setRedirect('multiple_email.manage', [
      'user' => (int) $this->email->getOwnerId(),
    ]);
  }

  /**
   * AJAX form submission handler.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    $this->doAction();

    $form = $this->formBuilder->getForm(ManageForm::class, $this->email->getOwner());
    $response = (new AjaxResponse())
      ->addCommand(new CloseModalDialogCommand())
      ->addCommand(new ReplaceCommand('#multiple-email-table-wrapper', $form));

    /** @var \Symfony\Component\HttpFoundation\Response $response */
    $form_state->setResponse($response);

    /** @var \Drupal\Core\Ajax\AjaxResponse $response */
    return $response;
  }

  /**
   * Closes the modal dialog.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response.
   */
  public function closeDialog(): AjaxResponse {
    return (new AjaxResponse())->addCommand(new CloseModalDialogCommand());
  }

  /**
   * Gets the description of the action this form is confirming.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The description of the action.
   */
  abstract protected function getDescription(): TranslatableMarkup;

  /**
   * Performs the action this form is confirming.
   */
  abstract protected function doAction();

}
