<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Form;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailStorageInterface;
use Drupal\multiple_email\Traits\MultipleEmailOperationsTrait;
use Drupal\user\UserInterface;

/**
 * Form for managing an accounts email addresses.
 *
 * @internal
 */
class ManageForm extends FormBase {

  use AutowireTrait;
  use MultipleEmailOperationsTrait {
    getOperations as protected;
  }

  /**
   * The email storage.
   *
   * @var \Drupal\multiple_email\EmailStorageInterface
   */
  protected readonly EmailStorageInterface $storage;

  /**
   * Constructs a new \Drupal\multiple_email\Form\ManageForm instance.
   *
   * @param \Drupal\multiple_email\EmailConfirmerInterface $emailConfirmer
   *   The email confirmer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The translation service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   An error happened when getting the entity storage.
   */
  public function __construct(
    protected readonly EmailConfirmerInterface $emailConfirmer,
    EntityTypeManagerInterface $entity_type_manager,
    TranslationInterface $string_translation,
    MessengerInterface $messenger,
  ) {
    $this->storage = $entity_type_manager->getStorage('multiple_email');

    $this->setStringTranslation($string_translation);
    $this->setMessenger($messenger);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'multiple_email_manage';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?UserInterface $user = NULL): array {
    $header = [
      'email' => $this->t('Email'),
      'status' => $this->t('Status'),
      'operations' => $this->t('Operations'),
    ];

    $form['emails'] = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No email addresses found.'),
      '#attributes' => [
        'id' => 'multiple-email-table-wrapper',
      ],
    ];

    if ($user) {
      $form_state->set('multiple_email_owner', $user);

      foreach ($this->storage->loadByUser($user) as $email) {
        $operations = $this->ajaxAttributes()->getOperations($email);
        $status_text = $email->getStatusLabel();

        $form['emails'][$email->id()] = [
          'email' => [
            '#type' => 'markup',
            '#markup' => $email->getEmail(),
          ],
          'status' => [
            '#type' => 'markup',
            '#markup' => $status_text,
          ],
          'operations' => [
            '#type' => 'operations',
            '#links' => $operations,
          ],
        ];
      }
    }

    $form['emails']['add'] = [
      'email' => [
        '#type' => 'email',
      ],
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Add a new email address'),
      ],
      'operations' => [],
    ];

    $action_url = Url::fromRoute('multiple_email.manage', [
      'user' => $user->id(),
    ]);

    $form['#action'] = $action_url->toString();
    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValue(['emails', 'add']);
    $email_address = $values['email'];

    if ($this->storage->loadByEmail($email_address)) {
      $message = $this->t('The email address has been already registered.', [
        '@email' => $email_address,
      ]);
      $form_state->setErrorByName('email', $message);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred while saving the email address entity.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $emails = $form_state->getValue('emails');
    $email = reset($emails);

    if (!empty($email)) {
      $email_address = $email['email'];
      /** @var \Drupal\multiple_email\EmailInterface $entity */
      $entity = $this->storage->create([
        'email' => $email_address,
      ]);

      /** @var \Drupal\user\UserInterface $user */
      if ($user = $form_state->get('multiple_email_owner')) {
        $entity->setOwner($user);
      }

      $entity->save();
      $this->emailConfirmer->confirm($entity);

      $this->messenger()
        ->addMessage($this->t('The email address @email has been added.', ['@email' => $email_address]));
    }

    $form_state->setRedirect('multiple_email.manage', [
      'user' => (int) $entity->getOwnerId(),
    ]);
  }

}
