<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Form builder for the admin settings page.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
class SettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * Constructs a new \Drupal\multiple_email\Form\SettingsForm instance.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly ModuleHandlerInterface $moduleHandler,
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    TranslationInterface $string_translation,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'multiple_email_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['multiple_email.settings', 'multiple_email.mail'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $token_help = $this->t('Available variables are: [site:name], [site:url], [site:login-url], [user:display-name], [user:account-name], [multiple_email:email], [multiple_email:confirm_code], [multiple_email:confirm_url], [multiple_email:confirm_deadline].');

    if ($token_enabled = $this->moduleHandler->moduleExists('token')) {
      $token_help = $this->t('The list of available tokens that can be used in emails is provided below.');
    }

    $form['hide_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Hide the email field'),
      '#description' => $this->t('Hides the email field when editing a user account'),
      '#options' => ['No', 'Yes'],
      '#config_target' => 'multiple_email.settings:hide_field',
    ];

    $form['edit_emails'] = [
      '#type' => 'select',
      '#title' => $this->t('Allow email editing'),
      '#description' => $this->t('Allows editing email addresses. It is equivalent to deleting and adding a new email address, as edited emails must be re-confirmed. If enabled, email addresses (excluding the primary email) may be edited via the multiple email tab.'),
      '#options' => ['No', 'Yes'],
      '#config_target' => 'multiple_email.settings:edit_emails',
    ];

    $form['confirm_attempts'] = [
      '#type' => 'number',
      '#title' => $this->t('Confirmation attempts'),
      '#description' => $this->t('The number of times a confirmation code can be used before a new one is generated. If set to 0, no new codes are sent after the first one.'),
      '#config_target' => 'multiple_email.settings:confirm.attempts',
    ];

    $form['confirm_deadline'] = [
      '#type' => 'number',
      '#title' => $this->t('Confirmation days'),
      '#description' => $this->t('The number of days before a confirmation code expires. If 0, confirmations do not expire.'),
      '#config_target' => 'multiple_email.settings:confirm.deadline',
    ];

    $form['email'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Emails'),
    ];

    $form['email_confirmation'] = [
      '#type' => 'details',
      '#title' => $this->t('Confirmation'),
      '#description' => $this->t('Customize the message sent when a new email is added.') . ' ' . $token_help,
      '#group' => 'email',
    ];

    $form['email_confirmation']['confirmation_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#config_target' => 'multiple_email.mail:confirmation.subject',
    ];

    $form['email_confirmation']['confirmation_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#config_target' => 'multiple_email.mail:confirmation.body',
    ];

    $form['email_expiration'] = [
      '#type' => 'details',
      '#title' => $this->t('Expiration'),
      '#description' => $this->t('Customize the message sent when an unconfirmed email address expires.') . ' ' . $token_help,
      '#group' => 'email',
    ];

    $form['email_expiration']['expiration_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#config_target' => 'multiple_email.mail:expiration.subject',
    ];

    $form['email_expiration']['expiration_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#config_target' => 'multiple_email.mail:expiration.body',
    ];

    if ($token_enabled) {
      $form['email']['token_tree'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => ['user', 'multiple_email'],
        '#show_restricted' => TRUE,
        '#show_nested' => FALSE,
        '#weight' => 90,
      ];
    }

    return $form;
  }

}
