<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Handler;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultAllowed;
use Drupal\Core\Access\AccessResultForbidden;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\multiple_email\EmailInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The access control handler for the email address entity.
 *
 * @internal
 */
class MultipleEmailAccessControlHandler extends EntityAccessControlHandler implements EntityHandlerInterface {

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    return new static(
      $entity_type,
      $container->get('string_translation'),
    );
  }

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Handler\MultipleEmailAccessControlHandler instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    TranslationInterface $string_translation,
  ) {
    parent::__construct($entity_type);
    $this->setStringTranslation($string_translation);
  }

  /**
   * Returns an access result for allowed access to the entity.
   *
   * It caches the result per permissions and per dependencies.
   *
   * @param mixed $dependencies
   *   The dependencies to add to the access result.
   *
   * @return \Drupal\Core\Access\AccessResultAllowed
   *   The access result.
   */
  protected function allowed(mixed ...$dependencies): AccessResultAllowed {
    $access = AccessResult::allowed()->cachePerPermissions();

    foreach ($dependencies as $dependency) {
      $access->addCacheableDependency($dependency);
    }

    return $access;
  }

  /**
   * Returns an access result for not allowed access to the entity.
   *
   * It caches the result per permissions and per dependencies.
   *
   * @param mixed $dependencies
   *   The dependencies to add to the access result.
   *
   * @return \Drupal\Core\Access\AccessResultForbidden
   *   The access result.
   */
  protected function forbidden(mixed ...$dependencies): AccessResultForbidden {
    $access = AccessResult::forbidden()->cachePerPermissions();

    foreach ($dependencies as $dependency) {
      $access->addCacheableDependency($dependency);
    }

    return $access;
  }

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {
    $operations = [
      'cancel',
      'confirm',
      'delete',
      'edit',
      'remove',
      'resend_confirmation',
      'set_primary',
      'update',
      'view',
    ];

    // Do not allow unknown operations.
    if (!in_array($operation, $operations)) {
      // Do not cache the result, which depends only on the operation.
      return AccessResult::forbidden(sprintf('%s is not a known operation.', $operation))
        ->setCacheMaxAge(0);
    }

    /** @var \Drupal\multiple_email\EmailInterface $entity */
    $is_admin = $account->hasPermission($this->entityType->getAdminPermission());
    $is_owner = $account->hasPermission('use multiple emails') && $entity->isOwner($account);

    if (!$is_admin && !$is_owner) {
      return $this->forbidden($account, $entity);
    }

    // Primary emails can only be viewed.
    if ($entity->isPrimary() && $operation !== 'view') {
      return $this->forbidden($account, $entity)
        ->setReason('Primary emails can only be viewed.');
    }

    return match ($operation) {
      'resend_confirmation' => $this->checkResendConfirmationAccess($entity, $account),
      'set_primary' => $this->checkSetPrimaryAccess($entity, $account),
      default => $this->allowed($account, $entity),
    };
  }

  /**
   * Checks access for resending a confirmation email.
   *
   * @param \Drupal\multiple_email\EmailInterface $entity
   *   The email address entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check access for.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function checkResendConfirmationAccess(EmailInterface $entity, AccountInterface $account): AccessResultInterface {
    if ($entity->isUnconfirmed()) {
      return $this->allowed($account, $entity);
    }

    return $this->forbidden($account, $entity)
      ->setReason('Confirmation emails cannot be resent for confirmed email address entities.');
  }

  /**
   * Checks access for setting the primary email address entity.
   *
   * @param \Drupal\multiple_email\EmailInterface $entity
   *   The email address entity.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check access for.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  protected function checkSetPrimaryAccess(EmailInterface $entity, AccountInterface $account): AccessResultInterface {
    // It can only be set as primary when confirmed, and it is not the primary
    // email.
    if ($entity->isConfirmed()) {
      if ($entity->isPrimary()) {
        return $this->forbidden($account, $entity)
          ->setReason('Confirmed email address entities which are already primary cannot be set as primary.');
      }

      return $this->allowed($account, $entity);
    }

    return $this->forbidden($account, $entity)
      ->setReason('Unconfirmed email address entities cannot be set as primary.');
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResultInterface {
    $is_admin = $account->hasPermission($this->entityType->getAdminPermission());
    $can_use_multiple_emails = $account->hasPermission('use multiple emails');
    $result = ($is_admin || $can_use_multiple_emails) ? 'allowed' : 'forbidden';

    return $this->$result($account);
  }

}
