<?php

declare(strict_types=1);

// phpcs:enable

namespace Drupal\multiple_email\Handler;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\multiple_email\Traits\MultipleEmailOperationsTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a class to build a collection of email address entities.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
class MultipleEmailListBuilder extends EntityListBuilder {

  use MultipleEmailOperationsTrait;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter'),
      $container->get('redirect.destination'),
      $container->get('string_translation'),
    );
  }

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Handler\MultipleEmailListBuilder instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $redirect_destination
   *   The redirect destination service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected readonly DateFormatterInterface $dateFormatter,
    RedirectDestinationInterface $redirect_destination,
    TranslationInterface $string_translation,
  ) {
    parent::__construct($entity_type, $storage);
    $this->setRedirectDestination($redirect_destination);
    $this->setStringTranslation($string_translation);
  }

  // phpcs:enable

  /**
   * {@inheritdoc}
   */
  public function load(): array {
    /** @var \Drupal\Core\Entity\EntityQueryInterface $entity_query */
    $entity_query = $this->storage->getQuery()
      ->accessCheck()
      ->pager(50);

    $header = $this->buildHeader();
    $entity_query->tableSort($header);

    $ids = $entity_query->execute();
    return $this->storage->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header = [
      'username' => [
        'data' => $this->t('Username'),
        'field' => 'uid',
        'specifier' => 'uid',
      ],
      'previous_owner' => [
        'data' => $this->t('Previous owner'),
        'field' => 'previous_owner',
        'specifier' => 'previous_owner',
      ],
      'email' => [
        'data' => $this->t('Email'),
        'field' => 'email',
        'specifier' => 'email',
      ],
      'created' => [
        'data' => $this->t('Created'),
        'field' => 'created',
        'specifier' => 'created',
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'confirmation_time' => [
        'data' => $this->t('Confirmation time'),
        'field' => 'confirmation_time',
        'specifier' => 'confirmation_time',
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'attempts' => [
        'data' => $this->t('Attempts'),
        'field' => 'attempts',
        'specifier' => 'attempts',
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'status' => [
        'data' => $this->t('Status'),
        'field' => 'status',
        'specifier' => 'status',
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
    ];

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function render(): array {
    $build = parent::render();
    $build['table']['#empty'] = $this->t('No email addresses have been added.');

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\multiple_email\EmailInterface $entity */
    $previous_owner = $entity->getPreviousOwner();
    $row['username']['data'] = [
      '#theme' => 'username',
      '#account' => $entity->getOwner(),
    ];

    if ($previous_owner) {
      if ($previous_owner->isAnonymous()) {
        $row['previous_owner']['data'] = $this->t('None');
      }
      elseif ($previous_owner->id() == $entity->getOwnerId()) {
        $row['previous_owner']['data'] = $this->t('Same user');
      }
      else {
        $row['previous_owner']['data'] = [
          '#theme' => 'username',
          '#account' => $previous_owner,
        ];
      }
    }
    else {
      $row['previous_owner']['data'] = $this->t('None');
    }

    $row['email'] = $entity->getEmail();
    $row['created'] = $this->dateFormatter->format($entity->getCreatedTime());
    $row['confirmation_time'] = $this->dateFormatter->format($entity->getConfirmationTime());
    $row['attempts'] = $entity->getAttempts();
    $row['status'] = $entity->getStatusLabel();

    return $row + parent::buildRow($entity);
  }

}
