<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\multiple_email\EmailConfirmerInterface;
use Drupal\multiple_email\EmailInterface;

/**
 * Cron hook implementations.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3566359
 */
class MultipleEmailCronHooks {

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Hook\MultipleEmailCronHooks instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\multiple_email\EmailConfirmerInterface $emailConfirmer
   *   The email confirmer service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EmailConfirmerInterface $emailConfirmer,
  ) {}

  // phpcs:enable

  /**
   * Implements hook_cron().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   An error occurred while loading the email address entity type.
   */
  #[Hook('cron')]
  public function cron(): void {
    $confirm = $this->configFactory->get('multiple_email.settings')->get('confirm');

    if ($confirm['deadline']) {
      $deadline_time = strtotime('-' . $confirm['deadline'] . ' days');
      $email_storage = $this->entityTypeManager->getStorage('multiple_email');
      $query = $email_storage->getQuery()
        ->accessCheck(FALSE)
        ->condition('status', EmailInterface::UNCONFIRMED)
        ->condition('time_code_generated', $deadline_time, '<');
      $ids = $query->execute();
      $emails = $email_storage->loadMultiple($ids);

      /** @var \Drupal\multiple_email\EmailInterface $email */
      foreach ($emails as $email) {
        $this->emailConfirmer->expire($email);
      }
    }
  }

}
