<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\multiple_email\EmailInterface;

/**
 * Entity hook implementations.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3566359
 */
class MultipleEmailEntityHooks {

  use StringTranslationTrait;

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Hook\MultipleEmailEntityHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    TranslationInterface $translation,
  ) {
    $this->setStringTranslation($translation);
  }

  // phpcs:enable

  /**
   * Implements hook_ENTITY_TYPE_delete().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   An error occurred while loading the email address entity type.
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred while deleting the email address entity.
   */
  #[Hook('user_delete')]
  public function userDelete(EntityInterface $entity): void {
    /** @var \Drupal\user\UserInterface $entity */
    /** @var \Drupal\multiple_email\EmailStorageInterface $email_storage */
    $email_storage = $this->entityTypeManager->getStorage('multiple_email');

    foreach ($email_storage->loadByUser($entity) as $email) {
      $email->delete();
    }
  }

  /**
   * Implements hook_ENTITY_TYPE_insert().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred while loading the email address entity type.
   */
  #[Hook('user_insert')]
  public function userInsert(EntityInterface $entity): void {
    /** @var \Drupal\user\UserInterface $entity */
    /** @var \Drupal\multiple_email\EmailStorageInterface $email_storage */
    $email_storage = $this->entityTypeManager->getStorage('multiple_email');
    $email = $email_storage->create([
      'uid' => $entity->id(),
      'email' => $entity->getEmail(),
      'status' => EmailInterface::CONFIRMED,
    ]);

    $email->save();
  }

  /**
   * Implements hook_ENTITY_TYPE_update().
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred while loading the email address entity type.
   */
  #[Hook('user_update')]
  public function userUpdate(EntityInterface $entity): void {
    /** @var \Drupal\user\UserInterface $entity */
    /** @var \Drupal\multiple_email\EmailStorageInterface $email_storage */
    $email_storage = $this->entityTypeManager->getStorage('multiple_email');
    $original_address = $entity->getOriginal()->getEmail();
    $new_address = $entity->getEmail();

    if ($original_address !== $new_address) {
      if ($original_email = $email_storage->loadByEmail($original_address)) {
        // Change the owner ID to 0 and save the email address entity. It could
        // be helpful in the case the entity is re-added to an account.
        $original_email->setOwnerToAnonymous()->save();
      }

      $new_email = $email_storage->create([
        'uid' => $entity->id(),
        'email' => $new_address,
        'status' => EmailInterface::CONFIRMED,
      ]);

      $new_email->save();
    }
  }

}
