<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;

/**
 * Form hook implementations.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3566359
 */
class MultipleEmailFormHooks {

  use StringTranslationTrait;

  // phpcs:disable Drupal.Files.LineLength.TooLong

  /**
   * Constructs a new \Drupal\multiple_email\Hook\MultipleEmailFormHooks instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The route match.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation
   *   The string translation service.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly AccountProxyInterface $currentUser,
    protected readonly RouteMatchInterface $routeMatch,
    TranslationInterface $translation,
  ) {
    $this->setStringTranslation($translation);
  }

  // phpcs:enable

  /**
   * Implements hook_form_BASE_FORM_ID_alter().
   */
  #[Hook('form_user_register_form_alter')]
  public function formUserRegisterFormAlter(array &$form, FormStateInterface $form_state): void {
    $form['account']['mail']['#element_validate'][] = static::class . ':validateUserRegisterForm';
  }

  /**
   * Validation handler for user registration forms.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   An error occurred while loading the email address entity type.
   */
  public function validateUserRegisterForm(array &$element, FormStateInterface $form_state): void {
    $email = $element['#value'];

    if (!empty($email)) {
      /** @var \Drupal\multiple_email\EmailStorageInterface $email_storage */
      $email_storage = $this->entityTypeManager->getStorage('multiple_email');

      if ($email_storage->loadByEmail($email)) {
        $message = $this->t('The email address %email is already registered.', [
          '%email' => $email,
        ]);

        $form_state->setError($element, $message);
      }
    }
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   *
   * Remove the email field from the profile edit form for privileged users.
   * This will be instead done in the address management screen.
   */
  #[Hook('form_user_form_alter')]
  public function formUserFormAlter(array &$form, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();

    if ($form_object instanceof EntityFormInterface) {
      /** @var \Drupal\Core\Entity\EntityInterface $entity */
      $entity = $form_object->getEntity();

      if ($entity->isNew()) {
        return;
      }
    }

    /** @var \Drupal\user\UserInterface $user */
    $user = $this->routeMatch->getParameter('user');
    $url = Url::fromRoute('multiple_email.manage', ['user' => $user->id()]);
    $access = $url->access();
    $hide_field = $this->configFactory
      ->get('multiple_email.settings')
      ->get('hide_field');

    if ($access && $hide_field) {
      $description = $this->t('Email addresses are managed on the <a href=":tab_url">Email addresses</a> tab.', [
        ':tab_url' => $url->toString(),
      ]);

      $form['account']['mail']['#disabled'] = TRUE;
      $form['account']['mail']['#required'] = FALSE;
      $form['account']['mail']['#description'] = $description;
    }
  }

}
