<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Plugin\views\relationship;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsRelationship;
use Drupal\views\Plugin\views\relationship\RelationshipPluginBase;

/**
 * Provides a view relationship.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
#[ViewsRelationship('multiple_email_relationship')]
class EmailViewsRelationship extends RelationshipPluginBase {

  /**
   * {@inheritdoc}
   */
  public function defineOptions(): array {
    $options = parent::defineOptions();
    $options['status'] = ['default' => [1]];
    $options['required'] = ['default' => TRUE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    $form['status'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Include'),
      '#options' => [
        0 => $this->t('Unconfirmed emails'),
        1 => $this->t('Confirmed emails'),
      ],
      '#default_value' => $this->getStatus(),
      '#required' => TRUE,
    ];

    // Every user account should always have at least an email address entity;
    // the required field is irrelevant.
    $form['required']['#access'] = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    $status = $this->getStatus();

    if (!empty($status)) {
      $this->definition['extra'][] = [
        'field' => 'status',
        'value' => $status,
        'numeric' => TRUE,
      ];
    }

    parent::query();
  }

  /**
   * Returns the status.
   *
   * @return array
   *   The status value.
   */
  protected function getStatus(): array {
    // Filter the selected options with is_string() to avoid excluding the
    // unconfirmed value, which is 0.
    return array_filter($this->options['status'], 'is_string');
  }

}
