<?php

declare(strict_types=1);

namespace Drupal\multiple_email\Traits;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Url;

/**
 * Trait to build the email address entity operations.
 *
 * @internal
 *
 * @see https://www.drupal.org/node/3565417
 */
trait MultipleEmailOperationsTrait {

  /**
   * Whether to use AJAX attributes for the operations.
   *
   * @var bool
   */
  protected bool $useAjaxAttributes = FALSE;

  /**
   * Adds AJAX attributes to the operations array.
   *
   * @param bool $useAjaxAttributes
   *   TRUE to add AJAX attributes to the operation array, FALSE otherwise.
   *
   * @return $this
   */
  protected function ajaxAttributes(bool $useAjaxAttributes = TRUE): static {
    $this->$useAjaxAttributes = $useAjaxAttributes;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(EntityInterface $entity): array {
    $ajax_attributes = [
      'class' => ['use-ajax'],
      'data-dialog-type' => 'modal',
      'data-dialog-options' => Json::encode([
        'width' => 'auto',
      ]),
    ];
    $attributes = $this->useAjaxAttributes ? $ajax_attributes : [];
    $parent_method = method_exists(get_parent_class($this), 'getOperations');

    /** @var \Drupal\multiple_email\EmailInterface $entity */
    // @phpstan-ignore staticMethod.notFound
    $operations = $parent_method ? parent::getOperations($entity) : [];

    if (isset($operations['edit'])) {
      $destination = $this->getRedirectDestination()->getAsArray();
      $operations['edit']['query'] = $destination;
    }

    // The 'View' operation doesn't make sense for email address entities.
    unset($operations['view']);

    if ($entity->isConfirmed()) {
      if (!$entity->isPrimary()) {
        $operations['set-primary'] = [
          'title' => $this->t('Set Primary'),
          'weight' => 10,
          'url' => Url::fromRoute(
            'multiple_email.set_primary',
            ['multiple_email' => $entity->id()],
          ),
          'attributes' => $attributes,
        ];
      }
    }
    elseif ($entity->isPending()) {
      $operations['resend'] = [
        'title' => $this->t('Resend confirmation'),
        'weight' => 15,
        'url' => Url::fromRoute(
          'multiple_email.resend',
          ['multiple_email' => $entity->id()],
        ),
        'attributes' => $attributes,
      ];
      $operations['cancel'] = [
        'title' => $this->t('Cancel confirmation'),
        'weight' => 20,
        'url' => Url::fromRoute(
          'multiple_email.cancel_confirmation',
          ['multiple_email' => $entity->id()],
        ),
        'attributes' => $attributes,
      ];
    }
    elseif ($entity->isCancelled()) {
      $operations['resend'] = [
        'title' => $this->t('Resend confirmation'),
        'weight' => 15,
        'url' => Url::fromRoute(
          'multiple_email.resend',
          ['multiple_email' => $entity->id()],
        ),
        'attributes' => $attributes,
      ];
    }
    else {
      $operations['confirm'] = [
        'title' => $this->t('Confirm'),
        'weight' => 5,
        'url' => Url::fromRoute(
          'multiple_email.confirm',
          ['multiple_email' => $entity->id()],
        ),
        'attributes' => $attributes,
      ];
    }

    // The primary email address cannot be removed.
    if (!$entity->isPrimary()) {
      $operations['remove'] = [
        'title' => $this->t('Remove'),
        'weight' => 25,
        'url' => Url::fromRoute(
          'multiple_email.remove',
          ['multiple_email' => $entity->id()],
        ),
        'attributes' => $attributes,
      ];
    }

    return $operations;
  }

}
