<?php

declare(strict_types=1);

namespace Drupal\Tests\multiple_email\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\multiple_email\Entity\Email;
use Drupal\multiple_email\Handler\MultipleEmailAccessControlHandler;
use Drupal\Tests\user\Traits\UserCreationTrait;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

/**
 * Tests the email address entity access handler.
 */
#[CoversClass(MultipleEmailAccessControlHandler::class)]
#[Group('multiple_email')]
#[RunTestsInSeparateProcesses]
/**
 * Tests the email address entity access handler.
 */
class MultipleEmailAccessControlHandlerTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'multiple_email',
    'system',
    'user',
  ];

  /**
   * The accounts used for the test.
   *
   * @var \Drupal\user\UserInterface[]
   */
  protected array $accounts;

  /**
   * The email address entity used for testing.
   *
   * @var \Drupal\multiple_email\Entity\Email
   */
  protected Email $emailEntity;

  /**
   * The access control handler to test.
   */
  protected MultipleEmailAccessControlHandler $accessHandler;

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   The entity type plugin for the email address entity was not found.
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurrent when creating the test accounts.
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('multiple_email');
    $this->installConfig('multiple_email');

    // Create first the account whose ID is 1, so other accounts do not have all
    // the permissions.
    $this->createUser([], values: ['uid' => 1]);

    $this->accounts[0] = $this->createUser(['administer multiple emails']);
    $this->accounts[1] = $this->createUser(['use multiple emails']);
    $this->accounts[2] = $this->createUser(['use multiple emails']);
    $this->accounts[3] = $this->createUser(['access content']);
    $this->emailEntity = Email::create(
      ['email' => $this->randomMachineName() . '@example.com'],
    );
    $this->accessHandler = new MultipleEmailAccessControlHandler(
      \Drupal::entityTypeManager()->getDefinition('multiple_email'),
      \Drupal::translation()
    );

    $this->emailEntity->save();
  }

  /**
   * Tests the operations available for an unconfirmed email address entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred when saving an entity.
   */
  public function testUnconfirmedEmail() {
    $unknown_operations = [
      $this->randomMachineName(),
      $this->randomMachineName(),
      $this->randomMachineName(),
      $this->randomMachineName(),
    ];

    $this->emailEntity->setOwner($this->accounts[1])
      ->setUnconfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isUnconfirmed(), 'The entity is unconfirmed.');

    // An account with the "use multiple emails" permission can access the
    // email address entities associated to it.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[1]), 'The account #1 can view the entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[1]), 'The account #1 can edit the entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[1]), 'The account #1 can delete the entity.');

    // An account without the "use multiple emails" nor the
    // "administer multiple emails" permission cannot access email address
    // entities associated to other accounts.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[3]), 'The account #3 cannot view the entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[3]), 'The account #3 cannot edit the entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[3]), 'The account #3 cannot delete the entity.');

    // An account with the "use multiple emails" permission cannot access
    // email address entities associated to other accounts.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[2]), 'The account #2 cannot view the entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[2]), 'The account #2 cannot edit the entity.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[2]), 'The account #2 cannot delete the entity.');

    // An account with the "administer multiple emails" permission can access
    // email address entities associated to other accounts.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'view', $this->accounts[0]), 'The account #0 can view the entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'edit', $this->accounts[0]), 'The account #0 can edit the entity.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'delete', $this->accounts[0]), 'The account #0 can delete the entity.');

    // A new confirmation email can only be requested by the owner and the admin
    // accounts.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[0]), 'The account #0 can request a new confirmation email.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[1]), 'The account #1 can request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[2]), 'The account #2 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[3]), 'The account #3 cannot request a new confirmation email.');

    // Unconfirmed email address entities cannot be set as primary email.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[0]), 'The account #0 cannot set the entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[1]), 'The account #1 cannot set the entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[2]), 'The account #2 cannot set the entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[3]), 'The account #3 cannot set the entity as primary.');

    // Unknown operations cannot be performed.
    foreach ($unknown_operations as $operation) {
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[1]), 'The account #1 cannot perform the "' . $operation . '" operation.');
    }
  }

  /**
   * Tests the operations available for a confirmed email address entity.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error occurred when saving an entity.
   */
  public function testConfirmedEmail() {
    $unknown_operations = [
      $this->randomMachineName(),
      $this->randomMachineName(),
    ];

    $this->emailEntity->setOwner($this->accounts[1])
      ->setConfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isConfirmed(), 'The entity is confirmed.');

    // A new confirmation email cannot be requested for confirmed email address
    // entities.
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[0]), 'The account #0 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[1]), 'The account #1 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[2]), 'The account #2 cannot request a new confirmation email.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'resend_confirmation', $this->accounts[3]), 'The account #3 cannot request a new confirmation email.');

    // Confirmed email address entities can be set as primary email.
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[0]), 'The account #0 can set the entity as primary.');
    $this->assertTrue($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[1]), 'The account #1 can set the entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[2]), 'The account #2 cannot set the entity as primary.');
    $this->assertFalse($this->accessHandler->access($this->emailEntity, 'set_primary', $this->accounts[3]), 'The account #3 cannot set the entity as primary.');

    // Unknown operations cannot be performed.
    foreach ($unknown_operations as $operation) {
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[1]), 'The account #1 cannot perform the "' . $operation . '" operation.');
    }
  }

  /**
   * Tests that the primary email can only be viewed.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   *   An error happened when saving an entity.
   */
  public function testPrimaryEmail() {
    $operations = [
      'cancel',
      'confirm',
      'delete',
      'edit',
      'remove',
      'resend_confirmation',
      'set_primary',
      'update',
    ];
    $tested_operation = 'view';
    $unknown_operations = [
      $this->randomMachineName(),
      $this->randomMachineName(),
      $this->randomMachineName(),
      $this->randomMachineName(),
    ];

    // Set the email as primary email for the account #1.
    $this->accounts[1]->setEmail($this->emailEntity->getEmail())->save();
    $this->emailEntity->setOwner($this->accounts[1])
      ->setConfirmed()
      ->save();
    $this->assertTrue($this->emailEntity->isOwner($this->accounts[1]), 'The entity owner has been set to the account #1.');
    $this->assertTrue($this->emailEntity->isConfirmed(), 'The entity is confirmed.');
    $this->assertTrue($this->emailEntity->isPrimary(), 'The entity is a primary email.');

    // Test that admin and owner account can view the email address entity.
    $this->assertTrue((bool) $this->accessHandler->access($this->emailEntity, $tested_operation, $this->accounts[0]), 'Account #0 can view the entity');
    $this->assertTrue((bool) $this->accessHandler->access($this->emailEntity, $tested_operation, $this->accounts[1]), 'Account #1 can view the entity');

    // Test that other operations are not allowed for any account.
    foreach ($operations as $operation) {
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[0]), sprintf('The %s operation is not allowed for the account #0', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[1]), sprintf('The %s operation is not allowed for the account #1', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[2]), sprintf('The %s operation is not allowed for the account #2', $operation));
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[3]), sprintf('The %s operation is not allowed for the account #3', $operation));
    }

    // Unknown operations cannot be performed.
    foreach ($unknown_operations as $operation) {
      $this->assertFalse($this->accessHandler->access($this->emailEntity, $operation, $this->accounts[1]), 'The account #1 cannot perform the "' . $operation . '" operation.');
    }
  }

  /**
   * Tests email address entities can be created by the right accounts.
   */
  public function testCreateAccess() {
    // Only accounts with either the "use multiple emails" or the
    // "administer multiple emails" permission can create email address
    // entities.
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[0]), 'The account #0 can create an email address entity.');
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[1]), 'The account #1 can create an email address entity.');
    $this->assertTrue($this->accessHandler->createAccess('multiple_email', $this->accounts[2]), 'The account #2 can create an email address entity.');
    $this->assertFalse($this->accessHandler->createAccess('multiple_email', $this->accounts[3]), 'The account #3 cannot create an email address entity.');
  }

}
