<?php

declare(strict_types=1);

namespace Drupal\multiselect_single\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsWidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the select widget.
 *
 * This widget provides individual select dropdowns for each delta
 * of a multivalue entity reference field, rather than a single
 * multi-select element.
 */
#[FieldWidget(
  id: 'multiselect_single',
  label: new TranslatableMarkup('Select list (multiple with individual dropdowns)'),
  field_types: [
    'entity_reference',
  ],
  multiple_values: FALSE,
)]
class MultiselectSingleWidget extends OptionsWidgetBase {

  /**
   * Builds a single-select version of the 'select' form element.
   *
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $field_definition = $this->fieldDefinition;
    $storage = $field_definition->getFieldStorageDefinition();
    $property = $storage->getMainPropertyName();
    $options = $this->getOptions($items->getEntity());

    $current = isset($items[$delta]) ? $items[$delta]->get($property)->getValue() : NULL;

    $element += [
      '#type' => 'select',
      '#title' => $field_definition->getLabel(),
      '#options' => $options,
      '#empty_option' => $this->t('- Select -'),
      '#empty_value' => '',
      '#default_value' => $current ?? '',
      '#required' => ($this->fieldDefinition->isRequired() && $delta === 0),
      '#element_validate' => [
        [static::class, 'validateElement'],
      ],
      '#multiple' => FALSE,
      '#error_no_message' => FALSE,
      '#attributes' => [
        'class' => [
          'form-element--type-select-individual',
        ],
      ],
    ];

    $element['#title'] = $this->t($field_definition->getLabel());
    return ['value' => $element];
  }

  /**
   * Element-level validation.
   */
  public static function validateElement(array $element, FormStateInterface $form_state): void {
    $value = $element['#value'] ?? '';

    if ($element['#required'] && $value === '') {
      $form_state->setError($element, t('@name field is required.', ['@name' => $element['#title']]));
    }

    $options = $element['#options'] ?? [];

    if ($value !== '' && !array_key_exists($value, $options)) {
      $form_state->setError($element, t('Invalid selection.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    $property = $this->fieldDefinition->getFieldStorageDefinition()->getMainPropertyName();
    $items = [];

    foreach ($values as $delta => $item) {
      $value = $item['value'] ?? '';
      $items[$delta] = [$property => ($value === '') ? NULL : $value];
    }

    return $items;
  }

}
