<?php

declare(strict_types=1);

namespace Drupal\myrest_seo\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

/**
 * Controller for safe redirects to related SEO admin pages.
 */
class MyrestSeoRedirectController extends ControllerBase {

  /**
   * Redirects to a target route if it exists; otherwise falls back.
   *
   * @param string $route_name
   *   The route machine name to redirect to.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the target or fallback route.
   */
  public function redirectPage(string $route_name): RedirectResponse {
    $provider = \Drupal::service('router.route_provider');

    // Try the provided route first; if это robots
    // из разных версий роботс-модуля.
    $candidates = [$route_name];
    if (str_starts_with($route_name, 'robotstxt')) {
      $candidates = array_unique(array_merge($candidates, [
        // Разные варианты названий роута, встречающиеся в 1.x ветках.
        'robotstxt.settings',
        'robotstxt_admin_settings',
        'robotstxt.admin_settings_form',
        'robotstxt.settings_form',
      ]));
    }

    foreach ($candidates as $candidate) {
      try {
        $provider->getRouteByName($candidate);
        // 302 — чтобы браузер не кэшировал админ-навигацию «насмерть».
        return $this->redirect($candidate, [], [], 302);
      }
      catch (RouteNotFoundException $e) {
        // Пробуем следующий кандидат.
      }
    }

    // Ничего не нашли — безопасный возврат на /admin/seo.
    return $this->redirect('myrest_seo', [], [], 302);
  }

  /**
   * Redirect to Robotstxt settings using a list of known route names.
   */
  public function robotstxt(): RedirectResponse {
    $candidates = [
      // Common route names across Robotstxt versions.
      'robotstxt.settings',
      'robotstxt_admin_settings',
      'robotstxt.admin_settings_form',
      'robotstxt.settings_form',
    ];
    return $this->redirectFirstExisting($candidates);
  }

  /**
   * Redirect to Redirect module list if available, otherwise fallback.
   */
  public function redirectList(): RedirectResponse {
    $candidates = [
      'redirect.list',
    ];
    return $this->redirectFirstExisting($candidates);
  }

  /**
   * Redirect to Simple Sitemap collection if available.
   */
  public function simpleSitemap(): RedirectResponse {
    $candidates = [
      'entity.simple_sitemap.collection',
      // Older route names can be added here if needed.
    ];
    return $this->redirectFirstExisting($candidates);
  }

  /**
   * Redirect to Metatag Defaults admin page (collection) if available.
   */
  public function metatagHub(): RedirectResponse {
    // Prefer custom collection if entity + storage table exist.
    if ($this->entityWithTableExists('myrest_metatag', 'myrest_metatag')) {
      return $this->redirectFirstExisting(['entity.myrest_metatag.collection']);
    }

    // If contrib Metatag admin exists, go there.
    $fallback = $this->redirectFirstExisting([
      'entity.metatag_defaults.collection',
      'metatag.admin',
    ]);

    // If neither custom nor contrib admin route is available, show a short
    // status page instead of silently bouncing back. This avoids 500 when
    // tables are missing and explains what to do.
    if ($fallback->getTargetUrl() === $this->redirect('myrest_seo', [], [], 302)->getTargetUrl()) {
      return $this->statusPageResponse(
        'Metatags by URL',
        'myrest_metatag',
        'myrest_metatag',
        'entity.myrest_metatag.collection'
      );
    }

    return $fallback;
  }

  /**
   * Redirect to custom "Template meta tags" list if available.
   *
   * This protects from REST service absence by falling back to the SEO
   * dashboard when the REST plugin manager is not available.
   */
  public function routeMetatagList(): RedirectResponse {
    // Prefer the custom collection provided by myrest_route_metatag when the
    // entity type and storage table are available.
    if ($this->entityWithTableExists('myrest_route_metatag', 'myrest_route_metatag')) {
      return $this->redirectFirstExisting(['entity.myrest_route_metatag.collection']);
    }

    // No destination available – show a small status page with instructions.
    return $this->statusPageResponse(
      'Metatag (routes/summary)',
      'myrest_route_metatag',
      'myrest_route_metatag',
      'entity.myrest_route_metatag.collection'
    );
  }

  /**
   * Tries a list of route names and redirects to the first existing one.
   *
   * @param string[] $route_names
   *   Candidate route machine names to try.
   */
  private function redirectFirstExisting(array $route_names): RedirectResponse {
    $provider = \Drupal::service('router.route_provider');
    foreach ($route_names as $candidate) {
      try {
        $provider->getRouteByName($candidate);
        return $this->redirect($candidate, [], [], 302);
      }
      catch (RouteNotFoundException $e) {
        // Try next candidate.
      }
    }
    return $this->redirect('myrest_seo', [], [], 302);
  }

  /**
   * Checks if the given entity type exists and its base table exists.
   */
  private function entityWithTableExists(string $entity_type_id, string $table): bool {
    try {
      if (!\Drupal::entityTypeManager()->hasDefinition($entity_type_id)) {
        return FALSE;
      }
    }
    catch (\Throwable $e) {
      return FALSE;
    }
    try {
      return \Drupal::database()->schema()->tableExists($table);
    }
    catch (\Throwable $e) {
      return FALSE;
    }
  }

  /**
   * Builds a small status page response with installation instructions.
   */
  private function statusPageResponse(string $title, string $module_name, string $table, string $target_route): RedirectResponse {
    // If a valid route exists after all, redirect there (race-condition safe).
    $provider = \Drupal::service('router.route_provider');
    try {
      $provider->getRouteByName($target_route);
      return $this->redirect($target_route, [], [], 302);
    }
    catch (RouteNotFoundException $e) {
      // Continue to message page.
    }

    // Compose a tiny message page and return a 302 back to SEO dashboard with
    // a user message (less intrusive than creating separate routes).
    $this->messenger()->addWarning($this->t('@title is not available yet. Ensure the module @module is enabled and database updates are run (drush updb).', [
      '@title' => $title,
      '@module' => $module_name,
    ]));
    return $this->redirect('myrest_seo', [], [], 302);
  }

}
