<?php

declare(strict_types=1);

namespace Drupal\myrest_seo\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Render\Markup;
use Drupal\Core\Cache\CacheableMetadata;

/**
 * Renders a simple summary table for Metatag Defaults.
 */
final class RouteMetatagListController extends ControllerBase {

  /**
   * Lists Metatag Defaults with quick edit links.
   *
   * @return array
   *   A render array with a table of defaults.
   */
  public function list(): array {
    $build = [
      '#type' => 'container',
    ];

    // Header.
    $header = [
      $this->t('Target'),
      $this->t('Applies to'),
      $this->t('Status'),
      $this->t('Operations'),
    ];

    $rows = [];

    // Load Metatag Defaults config entities if Metatag is enabled.
    if ($this->moduleHandler()->moduleExists('metatag')) {
      /** @var \Drupal\Core\Entity\EntityStorageInterface $storage */
      $storage = $this->entityTypeManager()->getStorage('metatag_defaults');
      $defaults = $storage->loadMultiple();

      foreach ($defaults as $default) {
        // Ensure scalar strings for strict analysis.
        $id = (string) $default->id();
        $label = (string) $default->label();
        $scope = 'global';
        if ($id !== '' && str_contains($id, '__')) {
          [$type, $bundle] = explode('__', $id, 2);
          $scope = $type . ':' . $bundle;
        }

        $values = method_exists($default, 'toArray') ? $default->toArray() : [];
        $status_text = !empty($values['tags']) ? $this->t('Configured') : $this->t('Empty');

        // Build edit link if the route exists.
        $edit_link = Link::fromTextAndUrl(
          $this->t('Edit'),
          Url::fromRoute('entity.metatag_defaults.edit_form', ['metatag_defaults' => $id])
        )->toRenderable();

        $rows[] = [
          ['data' => Markup::create($this->t('@label', ['@label' => $label])->__toString())],
          ['data' => ['#markup' => $scope]],
          ['data' => ['#markup' => $status_text]],
          ['data' => $edit_link],
        ];
      }
    }

    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No Metatag defaults found.'),
    ];

    // Cacheability: vary by language; invalidate on metatag defaults changes.
    $cache = new CacheableMetadata();
    $cache->setCacheContexts(['languages:language_interface']);
    // Generic tag to catch config changes of defaults.
    $cache->setCacheTags(['config:metatag_defaults']);
    $cache->applyTo($build);

    return $build;
  }

}
