<?php

declare(strict_types=1);

namespace Drupal\myrest_seo\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Applies simple SEO redirect rules early in the request.
 */
final class RedirectSubscriber implements EventSubscriberInterface {

  /**
   * Feature toggle and rules.
   */
  private ConfigFactoryInterface $configFactory;

  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run early, before controller resolution.
    // Higher priority executes earlier. 300 is sufficiently early for checks.
    return [
      KernelEvents::REQUEST => ['onKernelRequest', 300],
    ];
  }

  /**
   * Kernel request handler to apply redirect rules.
   */
  public function onKernelRequest(RequestEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();

    // Apply only to GET/HEAD to avoid interfering with POST/PUT/DELETE.
    if (!\in_array($request->getMethod(), ['GET', 'HEAD'], TRUE)) {
      return;
    }

    // Feature toggle.
    $settings = $this->configFactory->get('myrest_seo.settings');
    $enabled = (bool) $settings->get('redirects_enabled');

    // If feature toggle is disabled, skip.
    if (!$enabled) {
      return;
    }

    // Load rules.
    $redirects_config = $this->configFactory->get('myrest_seo.redirects');
    $rules = $redirects_config->get('rules') ?: [];
    if (empty($rules) || !\is_array($rules)) {
      return;
    }

    $path = $request->getPathInfo();
    // Normalize path: ensure leading slash, no trailing slash (except root).
    if ($path === '') {
      $path = '/';
    }

    foreach ($rules as $rule) {
      if (empty($rule['enabled'])) {
        continue;
      }
      $source = (string) ($rule['source'] ?? '');
      $target = (string) ($rule['target'] ?? '');
      $status = (int) ($rule['status'] ?? 301);

      if ($source === '' || $target === '') {
        continue;
      }

      // Exact path match for MVP. Optional regex may be added in future.
      if ($path !== $source) {
        continue;
      }

      // Prevent loops.
      if ($source === $target) {
        continue;
      }

      // Build absolute URL for internal targets starting with '/'.
      if (str_starts_with($target, '/')) {
        $url = Url::fromUserInput($target)->setAbsolute(TRUE)->toString();
      }
      else {
        $url = $target;
      }

      // Create cacheable redirect response.
      $response = new TrustedRedirectResponse($url, \in_array($status, [301, 302], TRUE) ? $status : 301);

      // Attach cacheability to the RESPONSE (not to a render array).
      // Vary by path and language; tag config; and prevent caching the
      // redirect.
      $bubble = new BubbleableMetadata();
      $bubble->setCacheContexts([
        'url.path',
        'languages:language_interface',
      ]);
      $bubble->setCacheTags(['config:myrest_seo.redirects']);
      $bubble->setCacheMaxAge(0);
      // Important: TrustedRedirectResponse implements
      // CacheableResponseInterface, so we must attach metadata via
      // addCacheableDependency(), not applyTo().
      $response->addCacheableDependency($bubble);

      $event->setResponse($response);
      return;
    }
  }

}
