<?php

declare(strict_types=1);

namespace Drupal\myrest_seo\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Admin form for managing simple redirect rules.
 */
final class RedirectsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'myrest_seo_redirects_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['myrest_seo.settings', 'myrest_seo.redirects'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $settings = $this->config('myrest_seo.settings');
    $redirects = $this->config('myrest_seo.redirects');

    // If the Redirect module is installed, warn about potential conflicts.
    if (\Drupal::moduleHandler()->moduleExists('redirect')) {
      $form['redirect_conflict_warning'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        '#markup' => $this->t('The <strong>Redirect</strong> module is enabled. MyREST SEO also applies redirect rules. Using both at the same time may cause duplicate or conflicting redirects. Prefer configuring redirects in one place. If you keep MyREST SEO redirects enabled, ensure your rules do not overlap with Redirect module rules.'),
        '#weight' => -100,
      ];
    }

    $form['redirects_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable MyREST SEO redirect rules'),
      '#default_value' => (bool) $settings->get('redirects_enabled'),
      '#description' => $this->t('When enabled, MyREST SEO applies simple redirect rules early in the request. Be careful if the Redirect module is also enabled — avoid duplicate rules.'),
    ];

    $rules = $redirects->get('rules') ?: [];
    $lines = [];
    foreach ($rules as $rule) {
      $lines[] = sprintf('%s|%s|%d|%d', (string) ($rule['source'] ?? ''), (string) ($rule['target'] ?? ''), (int) ($rule['status'] ?? 301), !empty($rule['enabled']) ? 1 : 0);
    }

    $form['rules'] = [
      '#type' => 'details',
      '#title' => $this->t('Redirect rules'),
      '#open' => TRUE,
    ];

    $form['rules']['rules_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Rules list'),
      '#default_value' => implode("\n", $lines),
      '#description' => $this->t('One rule per line, fields separated by a pipe (|): <code>source|target|status|enabled</code>. Example: <code>/old|/new|301|1</code>. Status: 301 or 302. Target may be an absolute URL or a site-relative path starting with a slash.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $text = (string) $form_state->getValue('rules_text');
    $raw_lines = preg_split('/\r?\n/', $text) ?: [];
    $lines = array_filter(array_map('trim', $raw_lines));
    $parsed = [];
    foreach ($lines as $index => $line) {
      $parts = array_map('trim', explode('|', $line));
      if (count($parts) !== 4) {
        $form_state->setErrorByName('rules_text', $this->t('Line @n: expected 4 fields separated by "|".', ['@n' => $index + 1]));
        continue;
      }
      [$source, $target, $status, $enabled] = $parts;
      if ($source === '' || $source[0] !== '/') {
        $form_state->setErrorByName('rules_text', $this->t('Line @n: source must start with "/".', ['@n' => $index + 1]));
      }
      $status_i = (int) $status;
      if (!in_array($status_i, [301, 302], TRUE)) {
        $form_state->setErrorByName('rules_text', $this->t('Line @n: status must be 301 or 302.', ['@n' => $index + 1]));
      }
      if ($target === '') {
        $form_state->setErrorByName('rules_text', $this->t('Line @n: target is required.', ['@n' => $index + 1]));
      }
      $enabled_b = (int) $enabled === 1 ? 1 : 0;
      $parsed[] = [
        'id' => md5($source . '|' . $target . '|' . $status_i),
        'source' => $source,
        'target' => $target,
        'status' => $status_i,
        'enabled' => (bool) $enabled_b,
      ];
    }
    // Store parsed result for submit.
    $form_state->setTemporaryValue('parsed_rules', $parsed);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $this->configFactory->getEditable('myrest_seo.settings')
      ->set('redirects_enabled', (bool) $form_state->getValue('redirects_enabled'))
      ->save();

    $parsed = (array) $form_state->getTemporaryValue('parsed_rules');
    $this->configFactory->getEditable('myrest_seo.redirects')
      ->set('rules', $parsed)
      ->save();
  }

}
