<?php

declare(strict_types=1);

namespace Drupal\myrest_seo\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Settings form for MyREST SEO.
 */
final class SeoSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'myrest_seo_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['myrest_seo.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('myrest_seo.settings');

    // Default to current host if not set.
    $default_front = (string) $config->get('robots_front_domain');
    if ($default_front === '') {
      $default_front = rtrim(\Drupal::request()->getSchemeAndHttpHost(), '/');
    }

    $form['robots_front_domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Front domain base URL'),
      '#description' => $this->t('Absolute base URL for the front application domain, e.g. https://example.com (no trailing slash). If empty, the current host will be used.'),
      '#default_value' => $default_front,
      '#required' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    $value = trim((string) $form_state->getValue('robots_front_domain'));
    if ($value !== '') {
      // Basic URL validation.
      if (!preg_match('@^https?://[^\s/$.?#].[^\s]*$@i', $value)) {
        $form_state->setErrorByName('robots_front_domain', $this->t('Please enter a valid URL starting with http:// or https://'));
      }
      // Remove trailing slash for consistency.
      $form_state->setValue('robots_front_domain', rtrim($value, '/'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $this->configFactory->getEditable('myrest_seo.settings')
      ->set('robots_front_domain', (string) $form_state->getValue('robots_front_domain'))
      ->save();
  }

}
