<?php

declare(strict_types=1);

/**
 * @file
 * Form alter callbacks for integrating with the Robotstxt admin form.
 */

namespace Drupal\myrest_seo\FormAlter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Alters Robotstxt admin settings form to add extra directives field.
 */
class RobotsTxtSettingsForm {

  /**
   * Adds/adjusts elements on the robots.txt admin form.
   *
   * @param array $form
   *   The form render array being built.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public static function alter(array &$form, FormStateInterface $form_state): void {
    // Admin site robots.txt URL (current host).
    $admin_url = Url::fromUserInput('/robots.txt')->setAbsolute()->toString();
    $form['robotstxt_content']['#title'] = '<a href="' . $admin_url . '" target="_blank">' . $admin_url . '</a>';
    $form['robotstxt_content']['#rows'] = 25;

    // Front site robots.txt URL (configurable front domain) — с fallback.
    $front_domain = (string) (\Drupal::config('myrest_seo.settings')->get('robots_front_domain') ?? '');
    if ($front_domain === '') {
      // Backward compatibility: fallback to legacy config, if present.
      $front_domain = (string) (\Drupal::config('myother.settings')->get('domain_app') ?? '');
    }
    $host = \Drupal::request()->getSchemeAndHttpHost();
    $domain_app_url = ($front_domain ? rtrim($front_domain, '/') : $host) . '/robots.txt';

    // robotstxt_content_for_domain_app.
    $robotstxt_config = \Drupal::configFactory()->get('robotstxt.settings');
    $form['robotstxt_content_for_domain_app'] = [
      '#type' => 'textarea',
      '#title' => '<a href="' . $domain_app_url . '" target="_blank">' . $domain_app_url . '</a>',
      '#default_value' => $robotstxt_config->get('robotstxt_content_for_domain_app'),
      "#rows" => 25,
    ];

    // Submit - наші дані зберігаємо 1ми (щоб кеш скинув базовий модуль)
    if (!isset($form['#submit']) || !is_array($form['#submit'])) {
      $form['#submit'] = [];
    }
    array_unshift($form['#submit'], [static::class, 'submit']);
  }

  /**
   * RobotsTxtSettingsForm submit.
   *
   * @param array $form
   *   The submitted form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state containing submitted values.
   */
  public static function submit(array &$form, FormStateInterface $form_state): void {
    $config = \Drupal::configFactory()->getEditable('robotstxt.settings');
    $config
      ->set('robotstxt_content_for_domain_app', $form_state->getValue('robotstxt_content_for_domain_app'))
      ->save();
  }

}
