<?php

namespace Drupal\nanobanana_editor\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\GenericType\ImageFile;
use Drupal\ai\OperationType\TextToImage\TextToImageInput;
use Drupal\ai_provider_nanobanana\OperationType\MultiImageToImageInput;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for generating images with NanoBanana AI.
 */
class ImageGeneratorForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new ImageGeneratorForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider plugin manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, AiProviderPluginManager $ai_provider_manager, FileSystemInterface $file_system) {
    $this->entityTypeManager = $entity_type_manager;
    $this->aiProviderManager = $ai_provider_manager;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('ai.provider'),
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nanobanana_image_generator_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#prefix'] = '<div id="image-generator-wrapper" class="nanobanana-editor-layout">';
    $form['#suffix'] = '</div>';

    // Main content container (left side).
    $form['main_content'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nanobanana-main-content']],
    ];

    // Image preview container.
    $form['main_content']['preview_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'image-preview-container', 'class' => ['nanobanana-preview']],
    ];

    // Check if we have a generated image in form state.
    $generated_image = $form_state->get('generated_image');
    if ($generated_image) {
      // Display the generated image as data URL.
      $data_uri = 'data:image/png;base64,' . base64_encode($generated_image);
      $form['main_content']['preview_container']['preview'] = [
        '#type' => 'html_tag',
        '#tag' => 'img',
        '#attributes' => [
          'src' => $data_uri,
          'alt' => $this->t('Generated image'),
          'class' => ['nanobanana-preview-image'],
        ],
      ];
    }
    else {
      $form['main_content']['preview_container']['placeholder'] = [
        '#markup' => '<div class="">' . $this->t('Your generated image will appear here.') . '</div>',
      ];
    }

    // Prompt textarea in main content.
    $form['main_content']['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt'),
      '#description' => $this->t('Describe the image you want to generate. Be specific and creative (e.g., "A serene mountain landscape at sunset with a lake in the foreground", "A futuristic city with flying cars").'),
      '#placeholder' => $this->t('E.g., A beautiful tropical beach with palm trees and crystal clear water'),
      '#required' => TRUE,
      '#rows' => 4,
    ];

    // Status messages container in main content.
    $form['main_content']['messages'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'generator-messages', 'class' => ['nanobanana-messages']],
    ];

    // Sidebar container (right side).
    $form['sidebar'] = [
      '#type' => 'fieldset',
      '#attributes' => ['class' => ['nanobanana-sidebar']],
    ];

    // Model selection field in sidebar.
    $form['sidebar']['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#description' => $this->t('Select the AI model to use for image generation.'),
      '#options' => [
        'gemini-2.5-flash-image' => $this->t('Gemini 2.5 Flash (Fast, supports up to 3 images)'),
        'gemini-3-pro-image-preview' => $this->t('Gemini 3 Pro (High quality, supports up to 14 images)'),
      ],
      '#default_value' => 'gemini-2.5-flash-image',
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateModelDependentFieldsCallback',
        'wrapper' => 'model-dependent-fields-wrapper',
        'event' => 'change',
      ],
    ];

    // Get the selected model (from form state or default).
    $selected_model = $form_state->getValue('model') ?: 'gemini-2.5-flash-image';
    $max_files = $selected_model === 'gemini-3-pro-image-preview' ? 14 : 3;
    $model_name = $selected_model === 'gemini-3-pro-image-preview' ? 'Gemini 3 Pro' : 'Gemini 2.5 Flash';
    $is_gemini_3_pro = $selected_model === 'gemini-3-pro-image-preview';

    // Model-dependent fields wrapper (with AJAX wrapper) in sidebar.
    $form['sidebar']['model_dependent_fields_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'model-dependent-fields-wrapper'],
    ];

    // Image upload field for multi-image composition.
    $form['sidebar']['model_dependent_fields_wrapper']['additional_images'] = [
      '#type' => 'file',
      '#title' => $this->t('Images (Optional)'),
      '#description' => $this->t('Upload one or more images. You can select multiple files at once (hold Ctrl/Cmd while clicking). For text-to-image generation, leave empty. For image-to-image or multi-image composition, upload images. @model supports up to @max images.', [
        '@model' => $model_name,
        '@max' => $max_files,
      ]),
      '#multiple' => TRUE,
      '#name' => 'files[additional_images][]',
      '#attributes' => [
        'accept' => 'image/png,image/jpeg,image/jpg',
        'multiple' => 'multiple',
      ],
      '#attached' => [
        'library' => ['nanobanana_editor/image_preview'],
      ],
      '#required' => FALSE,
    ];

    // Image Size field (only for Gemini 3 Pro).
    if ($is_gemini_3_pro) {
      $form['sidebar']['model_dependent_fields_wrapper']['image_size'] = [
        '#type' => 'select',
        '#title' => $this->t('Image Size'),
        '#description' => $this->t('Output image size (Gemini 3 Pro only). Higher resolution takes longer to generate.'),
        '#options' => [
          '' => $this->t('Default'),
          '1K' => $this->t('1K'),
          '2K' => $this->t('2K'),
          '4K' => $this->t('4K'),
        ],
        '#default_value' => '',
        '#required' => FALSE,
      ];
    }

    // Aspect ratio select field in sidebar.
    $form['sidebar']['aspect_ratio'] = [
      '#type' => 'select',
      '#title' => $this->t('Aspect Ratio'),
      '#description' => $this->t('Select the aspect ratio for the generated image.'),
      '#options' => [
        '1:1' => $this->t('1:1 (Square)'),
        '3:4' => $this->t('3:4 (Portrait)'),
        '4:3' => $this->t('4:3 (Landscape)'),
        '9:16' => $this->t('9:16 (Vertical)'),
        '16:9' => $this->t('16:9 (Widescreen)'),
      ],
      '#default_value' => '4:3',
      '#required' => TRUE,
    ];

    // Styles select field in sidebar.
    $config = \Drupal::config('nanobanana_editor.settings');
    $styles = $config->get('styles') ?: [];
    $style_options = ['' => $this->t('- None -')];
    foreach ($styles as $index => $style) {
      $style_options[$index] = $style['name'];
    }

    $form['sidebar']['style'] = [
      '#type' => 'select',
      '#title' => $this->t('Style'),
      '#description' => $this->t('Select a style to apply to the generated image. The style prompt will be added to your prompt.'),
      '#options' => $style_options,
      '#default_value' => '',
      '#required' => FALSE,
    ];

    // Media name field in sidebar.
    $form['sidebar']['media_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Media Name'),
      '#description' => $this->t('Name for the media item (optional - will be auto-generated if empty).'),
      '#required' => FALSE,
    ];

    // Footer
    $form['footer'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['nanobanana-footer']],
    ];

    // Actions in footer.
    $form['footer']['actions'] = [
      '#type' => 'actions',
    ];

    $form['footer']['actions']['generate'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate Image'),
      '#ajax' => [
        'callback' => '::generateCallback',
        'wrapper' => 'image-generator-wrapper',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating image... This may take up to 2 minutes.'),
        ],
      ],
    ];

    // Only show save button if we have a generated image.
    if ($generated_image) {
      $form['footer']['actions']['save'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save as Media'),
        '#submit' => ['::saveSubmit'],
      ];
    }

    $form['footer']['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('entity.media.collection'),
      '#attributes' => ['class' => ['button']],
    ];



    // Attach the library.
    $form['#attached']['library'][] = 'nanobanana_editor/editor';

    // Add validation handler.
    $form['#validate'][] = '::validateAdditionalImages';

    return $form;
  }

  /**
   * AJAX callback to update model-dependent fields.
   */
  public function updateModelDependentFieldsCallback(array &$form, FormStateInterface $form_state) {
    return $form['sidebar']['model_dependent_fields_wrapper'];
  }

  /**
   * Validation handler to check the number of additional images.
   */
  public function validateAdditionalImages(array &$form, FormStateInterface $form_state) {
    // Only validate when the generate button is clicked, not on model change.
    $triggering_element = $form_state->getTriggeringElement();
    if (isset($triggering_element['#name']) && $triggering_element['#name'] !== 'op') {
      // This is an AJAX callback for model change, skip validation.
      return;
    }

    // Check if additional images were uploaded.
    if (!isset($_FILES['files']['name']['additional_images']) || !is_array($_FILES['files']['name']['additional_images'])) {
      return;
    }

    // Count valid uploaded files.
    $file_count = 0;
    $file_errors = $_FILES['files']['error']['additional_images'];

    foreach ($file_errors as $error_data) {
      // Handle extra level of nesting.
      $error = is_array($error_data) && isset($error_data[0]) ? $error_data[0] : $error_data;

      if ($error === UPLOAD_ERR_OK) {
        $file_count++;
      }
    }

    if ($file_count > 0) {
      // Get the selected model.
      $model = $form_state->getValue('model');
      if (empty($model)) {
        return;
      }

      $max_files = $model === 'gemini-3-pro-image-preview' ? 14 : 3;

      \Drupal::logger('nanobanana_editor')->info('Validating: @count files uploaded, max allowed: @max', [
        '@count' => $file_count,
        '@max' => $max_files,
      ]);

      if ($file_count > $max_files) {
        $model_name = $model === 'gemini-3-pro-image-preview' ? 'Gemini 3 Pro' : 'Gemini 2.5 Flash';
        $form_state->setErrorByName('model_dependent_fields_wrapper][additional_images', $this->t('You uploaded @count images, but @model only supports up to @max images. Please select fewer files.', [
          '@count' => $file_count,
          '@model' => $model_name,
          '@max' => $max_files,
        ]));
      }
    }
  }

  /**
   * AJAX callback for the Generate button.
   */
  public function generateCallback(array &$form, FormStateInterface $form_state) {
    \Drupal::logger('nanobanana_editor')->info('Image generator callback started');

    $response = new AjaxResponse();

    // Check if there are validation errors.
    if ($form_state->hasAnyErrors()) {
      // Return the form with errors displayed.
      $errors = $form_state->getErrors();
      $error_message = implode('<br>', $errors);
      $response->addCommand(new HtmlCommand('#generator-messages', '<div class="messages messages--error">' . $error_message . '</div>'));
      return $response;
    }

    $prompt = $form_state->getValue('prompt');
    $aspect_ratio = $form_state->getValue('aspect_ratio');
    $style_index = $form_state->getValue('style');

    // Get configuration.
    $config = \Drupal::config('nanobanana_editor.settings');

    // Start with system instructions if available.
    $final_prompt = '';
    $system_instructions = $config->get('system_instructions');
    if (!empty($system_instructions)) {
      $final_prompt = $system_instructions . ' ';
      \Drupal::logger('nanobanana_editor')->info('Applied system instructions');
    }

    // Add user prompt.
    $final_prompt .= $prompt;

    // Merge style prompt if a style is selected.
    if ($style_index !== '') {
      $styles = $config->get('styles') ?: [];
      if (isset($styles[$style_index])) {
        $style_prompt = $styles[$style_index]['prompt'];
        $final_prompt .= ' ' . $style_prompt;
        \Drupal::logger('nanobanana_editor')->info('Applied style: @style', ['@style' => $styles[$style_index]['name']]);
      }
    }

    \Drupal::logger('nanobanana_editor')->info('Final prompt: @prompt, Aspect Ratio: @ratio', [
      '@prompt' => $final_prompt,
      '@ratio' => $aspect_ratio,
    ]);

    // Use the final merged prompt.
    $prompt = $final_prompt;

    try {
      // Load the AI provider.
      $provider = $this->aiProviderManager->createInstance('nanobanana');

      // Set the configuration with selected aspect ratio and imageSize.
      $configuration = [
        'aspectRatio' => $aspect_ratio,
      ];

      // Add imageSize if provided (Gemini 3 Pro only).
      $image_size = $form_state->getValue('image_size');
      if (!empty($image_size)) {
        $configuration['imageSize'] = $image_size;
      }

      $provider->setConfiguration($configuration);

      // Collect all uploaded images from multi-upload field.
      $image_files = [];

      if (isset($_FILES['files']['name']['additional_images'])) {
        $file_data = $_FILES['files']['name']['additional_images'];

        if (is_array($file_data)) {
          // Iterate through the uploaded files.
          foreach ($file_data as $index => $file_name_data) {
            $tmp_name_data = $_FILES['files']['tmp_name']['additional_images'][$index] ?? NULL;
            $file_type_data = $_FILES['files']['type']['additional_images'][$index] ?? NULL;
            $error_data = $_FILES['files']['error']['additional_images'][$index] ?? NULL;

            // Handle extra level of nesting.
            if (is_array($file_name_data) && isset($file_name_data[0])) {
              $file_name = $file_name_data[0];
              $tmp_name = $tmp_name_data[0] ?? NULL;
              $file_type = $file_type_data[0] ?? NULL;
              $error = $error_data[0] ?? NULL;
            }
            else {
              $file_name = $file_name_data;
              $tmp_name = $tmp_name_data;
              $file_type = $file_type_data;
              $error = $error_data;
            }

            if ($error === UPLOAD_ERR_OK && !empty($tmp_name) && is_string($tmp_name) && file_exists($tmp_name)) {
              $raw_file = file_get_contents($tmp_name);
              $image_files[] = new ImageFile($raw_file, $file_type, $file_name);
            }
          }
        }
      }

      // Get the selected model.
      $model = $form_state->getValue('model') ?: 'gemini-2.5-flash-image';

      // Determine which type of generation to use.
      if (count($image_files) > 1) {
        // Multi-image composition.
        $input = new MultiImageToImageInput($image_files[0]);
        for ($i = 1; $i < count($image_files); $i++) {
          $input->addAdditionalImage($image_files[$i]);
        }
        $input->setPrompt($prompt);
        \Drupal::logger('nanobanana_editor')->info('Using @count images for composition with model: @model', ['@count' => count($image_files), '@model' => $model]);
        $output = $provider->imageToImage($input, $model, ['nanobanana_editor']);
      }
      elseif (count($image_files) === 1) {
        // Single image to image.
        $input = new \Drupal\ai\OperationType\ImageToImage\ImageToImageInput($image_files[0]);
        $input->setPrompt($prompt);
        \Drupal::logger('nanobanana_editor')->info('Using single image generation with model: @model', ['@model' => $model]);
        $output = $provider->imageToImage($input, $model, ['nanobanana_editor']);
      }
      else {
        // Text to image.
        $input = new TextToImageInput($prompt);
        \Drupal::logger('nanobanana_editor')->info('Using text-to-image generation with model: @model', ['@model' => $model]);
        $output = $provider->textToImage($input, $model, ['nanobanana_editor']);
      }

      \Drupal::logger('nanobanana_editor')->info('AI provider returned response');

      // Get the first generated image.
      $images = $output->getNormalized();
      \Drupal::logger('nanobanana_editor')->info('Got @count images', ['@count' => count($images)]);

      if (!empty($images)) {
        $generated_image = $images[0]->getBinary();
        \Drupal::logger('nanobanana_editor')->info('Generated image size: @size bytes', ['@size' => strlen($generated_image)]);

        // Store the generated image in form state.
        $form_state->set('generated_image', $generated_image);

        // Rebuild the form.
        $form_state->setRebuild(TRUE);

        // Get the rebuilt form.
        $rebuilt_form = \Drupal::formBuilder()->rebuildForm($this->getFormId(), $form_state, $form);

        // Replace the entire form.
        $response->addCommand(new HtmlCommand('#image-generator-wrapper', $rebuilt_form));
        \Drupal::logger('nanobanana_editor')->info('Sending AJAX response with rebuilt form');
      }
      else {
        throw new \Exception('No image was generated.');
      }
    }
    catch (\Exception $e) {
      // Log the error.
      \Drupal::logger('nanobanana_editor')->error('Error generating image: @message', ['@message' => $e->getMessage()]);

      // Show error message.
      $error_message = $e->getMessage();

      // Provide more helpful guidance.
      if (strpos($error_message, 'NO_IMAGE') !== FALSE) {
        $error_message = $this->t('The AI did not generate an image. Try being more specific in your prompt. For example: "A mountain landscape with a blue sky" or "A modern office interior with large windows".');
      }

      $response->addCommand(new HtmlCommand('#generator-messages', '<div class="messages messages--error">' . $error_message . '</div>'));
    }

    return $response;
  }

  /**
   * Submit handler for the Save button.
   */
  public function saveSubmit(array &$form, FormStateInterface $form_state) {
    $generated_image = $form_state->get('generated_image');

    if (!$generated_image) {
      $this->messenger()->addError($this->t('No generated image to save.'));
      return;
    }

    try {
      // Create a file entity from the generated image.
      $directory = 'public://nanobanana-generated';
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

      // Generate filename.
      $media_name = $form_state->getValue('media_name');
      if (empty($media_name)) {
        $media_name = 'AI Generated Image ' . date('Y-m-d H:i:s');
      }
      $filename = preg_replace('/[^a-z0-9]+/', '-', strtolower($media_name)) . '-' . time() . '.png';
      $destination = $directory . '/' . $filename;

      // Save the image file.
      $file_uri = $this->fileSystem->saveData($generated_image, $destination, FileSystemInterface::EXISTS_RENAME);

      if ($file_uri) {
        // Create a file entity.
        $file = $this->entityTypeManager->getStorage('file')->create([
          'uri' => $file_uri,
          'status' => 1,
          'uid' => \Drupal::currentUser()->id(),
        ]);
        $file->save();

        // Create a media entity.
        $media = $this->entityTypeManager->getStorage('media')->create([
          'bundle' => 'image',
          'name' => $media_name,
          'field_media_image' => [
            'target_id' => $file->id(),
            'alt' => $media_name,
          ],
          'uid' => \Drupal::currentUser()->id(),
          'status' => 1,
        ]);
        $media->save();

        $this->messenger()->addStatus($this->t('Image saved as media: <a href="@url">@name</a>', [
          '@url' => $media->toUrl('edit-form')->toString(),
          '@name' => $media->label(),
        ]));

        // Redirect to the media edit form.
        $form_state->setRedirect('entity.media.edit_form', ['media' => $media->id()]);
      }
      else {
        throw new \Exception('Failed to save the generated image file.');
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error saving image: @message', ['@message' => $e->getMessage()]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This is handled by AJAX callbacks or specific submit handlers.
  }

}
