<?php

namespace Drupal\nanobanana_editor\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Configure NanoBanana settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'nanobanana_editor.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'nanobanana_editor_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure settings for NanoBanana AI image generation and editing.') . '</p>',
    ];

    // System instructions.
    $form['system_instructions'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Instructions'),
      '#description' => $this->t('Global instructions that will be prepended to all prompts. Use this to set consistent behavior, tone, or quality standards for all generated images.'),
      '#default_value' => $config->get('system_instructions') ?? '',
      '#rows' => 5,
    ];

    // Styles section.
    $form['styles_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Image Styles'),
      '#description' => $this->t('Manage image styles that users can select when generating or editing images.'),
    ];

    $styles = $config->get('styles') ?? [];

    if (!empty($styles)) {
      $form['styles_section']['styles_table'] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Name'),
          $this->t('Prompt'),
          $this->t('Operations'),
        ],
        '#empty' => $this->t('No styles configured yet.'),
      ];

      foreach ($styles as $index => $style) {
        $form['styles_section']['styles_table'][$index]['name'] = [
          '#markup' => $style['name'],
        ];

        $form['styles_section']['styles_table'][$index]['prompt'] = [
          '#markup' => $style['prompt'],
        ];

        $form['styles_section']['styles_table'][$index]['operations'] = [
          '#type' => 'operations',
          '#links' => [
            'edit' => [
              'title' => $this->t('Edit'),
              'url' => Url::fromRoute('nanobanana_editor.style_edit', ['style_index' => $index]),
            ],
            'delete' => [
              'title' => $this->t('Delete'),
              'url' => Url::fromRoute('nanobanana_editor.style_delete', ['style_index' => $index]),
            ],
          ],
        ];
      }
    }
    else {
      $form['styles_section']['no_styles'] = [
        '#markup' => '<p>' . $this->t('No styles configured yet.') . '</p>',
      ];
    }

    $form['styles_section']['add_style'] = [
      '#type' => 'link',
      '#title' => $this->t('Add style'),
      '#url' => Url::fromRoute('nanobanana_editor.style_add'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save system instructions.
    $this->config(static::CONFIG_NAME)
      ->set('system_instructions', $form_state->getValue('system_instructions'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
