((Drupal) => {
  /**
   * Handles image preview functionality for file uploads.
   *
   * @param {Event} event - The change event from file input.
   * @param {HTMLElement} previewContainer - The container for previews.
   */
  function handleFileChange(event, previewContainer) {
    // Clear previous previews.
    previewContainer.innerHTML = '';

    const files = event.target.files;
    if (files.length === 0) {
      return;
    }

    // Show count.
    const countLabel = document.createElement('div');
    countLabel.className = 'nanobanana-preview-count';
    const imageText = files.length > 1 ? 's' : '';
    countLabel.textContent = `${files.length} image${imageText} selected:`;
    previewContainer.appendChild(countLabel);

    // Create thumbnails wrapper.
    const thumbsWrapper = document.createElement('div');
    thumbsWrapper.className = 'nanobanana-preview-thumbnails';
    previewContainer.appendChild(thumbsWrapper);

    // Generate preview for each file.
    Array.from(files).forEach((file, index) => {
      if (!file.type.match('image.*')) {
        return;
      }

      const reader = new FileReader();

      reader.onload = (e) => {
        const thumbContainer = document.createElement('div');
        thumbContainer.className = 'nanobanana-preview-thumb';

        const img = document.createElement('img');
        img.src = e.target.result;
        img.alt = file.name;
        img.title = file.name;

        const label = document.createElement('div');
        label.className = 'nanobanana-preview-label';
        label.textContent = `${index + 1}. ${file.name}`;

        thumbContainer.appendChild(img);
        thumbContainer.appendChild(label);
        thumbsWrapper.appendChild(thumbContainer);
      };

      reader.readAsDataURL(file);
    });
  }

  Drupal.behaviors.nanoBananaImagePreview = {
    attach(context) {
      // Find all file inputs with 'additional_images' in the name.
      const fileInputs = context.querySelectorAll('input[type="file"]');

      fileInputs.forEach((fileInput) => {
        if (!fileInput.name || !fileInput.name.includes('additional_images')) {
          return;
        }

        // Only attach once.
        if (fileInput.dataset.nanobananaPreviewed) {
          return;
        }
        fileInput.dataset.nanobananaPreviewed = 'true';

        // Create or find preview container.
        let previewContainer = fileInput.parentElement.querySelector(
          '.nanobanana-image-previews',
        );
        if (!previewContainer) {
          previewContainer = document.createElement('div');
          previewContainer.className = 'nanobanana-image-previews';
          fileInput.parentElement.insertBefore(
            previewContainer,
            fileInput.nextSibling,
          );
        }

        // Handle file selection.
        fileInput.addEventListener('change', (e) => {
          handleFileChange(e, previewContainer);
        });
      });
    },
  };
})(Drupal);
